;; #############################################################################
;; # ========================================================================= #
;; # Bot IRC Syd - utils/convert-fortune-fr-files.ss                           #
;; # Copyright (C) Lozi Jean-Pierre, 2007 - mailto:jean-pierre@lozi.org        #
;; # ========================================================================= #
;; #                                                                           #
;; # This program is free software; you can redistribute it and/or modify it   #
;; # under the terms of the GNU General Public License as published by the     #
;; # Free Software Foundation; either version 2 of the License, or (at your    #
;; # option) any later version.                                                #
;; #                                                                           #
;; # This program is distributed in the hope that it will be useful, but WITH- #
;; # OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or # 
;; # FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for #
;; # for more details.                                                         #
;; #                                                                           #
;; # You should have received a copy of the GNU General Public License along   #
;; # with this program; if not, write to the Free Software Foundation, Inc.,   #
;; # 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.                 #
;; #                                                                           #
;; #############################################################################

;;
;; On a besoin de ce fichier pour manipuler les ports d'entrée et de sortie.
;;
(require (lib "port.ss"))

;;
;; Cette fonction permet de convertir facilement les citations du paquetage fortunes-fr (v0.2) en des citations pour le bot
;; Syd.
;;
(define (convert-fortune-fr-files fortunes-directory quotes-directory)
  ;; On affiche un message pour prévenir l'utilisateur.
  (display "Conversion en cours...")
  ;; On traite...
  (for-each
   ;; ...chaque fichier.
   (lambda (file)
     ;; On ouvre le fichier de sortie, en utilisant l'encoding ISO-8859-15, standard de facto sur irc (français).
     (let ((output-port (reencode-output-port (open-output-file (build-path quotes-directory file) 'replace) "ISO-8859-15")))
       ;; On sauve le contenu du fichier d'entrée en l'ouvrant...
       (let ((contents (let ((input-port (reencode-input-port (open-input-file (build-path fortunes-directory file))
                                                              ;; ...en utilisant encore l'encoding "ISO-8859-15"
                                                              "ISO-8859-15")))
                         ;; ...puis...
                         (begin0 
                           ;; ...en bouclant sur chaque ligne du fichier. On part avec un contenu vide.
                           (let loop ((file-contents ""))
                             ;; Puis, on lit une ligne.
                             (let ((current-line (read-line input-port)))
                               ;; Si on a atteint la fin du fichier...
                               (if (eof-object? current-line)
                                   ;; ...on renvoie le contenu.
                                   file-contents
                                   ;; Sinon, on continue à boucler, en ajoutant ce qu'on a lu à la fin du fichier. On perd
                                   ;; les retours à la ligne, car on veut que toutes les citations tiennent sur une ligne.
                                   ;; On remplace les lignes contenant seulement le caractère % par le caractère @, pour
                                   ;; éviter les interférences avec le caractère %, utilisé au milieu de certaines 
                                   ;; citations. @ n'est utilisé nulle part dans le paquetage fortunes-fr.
                                   (loop (string-append file-contents " " (regexp-replace* "^%$" current-line "@"))))))
                           ;; On prévient l'utilisateur de l'avancement...
                           (display ".")
                           ;; Enfin, on n'oublie pas de fermer le fichier.
                           (close-input-port input-port)))))
         ;; Puis on fait passer notre fichier à travers plusieurs expressions régulières. On utilise set! au lieu d'un let* 
         ;; pour éviter de surcharger inutilement la mémoire. D'abord, on ajoute des slashes avant les slashes du fichier
         ;; car ce sont des caractères spéciaux.
         (set! contents (regexp-replace* "\\\\" contents "\\\\\\"))
         ;; Ensuite, on remplace les " par des ' pour éviter les interférences avec nos guillemets.
         (set! contents (regexp-replace* "\"" contents "'"))
         ;; Les @ marquent les séparations entre citations. On les transforme en des fins de lignes, en ajoutant les bonnes
         ;; parenthèses.
         (set! contents (regexp-replace* "@ " contents "\")\n (\""))
         ;; On récupère les chaînes marquant le début de la source des citations, et on les transforme en guillemets, qui
         ;; sépareront les arguments des sous-listes.
         (set! contents (regexp-replace* " \t-?[+]- " contents "\" \""))
         ;; On supprime les chaînes marquant la fin des citations.
         (set! contents (regexp-replace* " -[+]- " contents ""))
         ;; On supprime les éventuelles tabulations superflues.
         (set! contents (regexp-replace* "\t" contents ""))
         ;; On efface les lignes contenant >, car ce caractère est généralement caractéristique d'une citation multi-lignes, 
         ;; qui passerait mal sur une seule ligne.
         (set! contents (regexp-replace* "\n[^\n]*>[^\n@]*" contents ""))
         ;; On efface les éventuels espaces superflus (espaces oubliés e nfin de ligne).
         (set! contents (regexp-replace* "[ ]+" contents " "))
         ;; On efface les espaces existants à la fin de certaines citations.
         (set! contents (regexp-replace* " \" \"" contents "\" \""))
         ;; S'il n'y a pas d'auteur dans la citation, on a un espace superflu, qu'on efface.
         (set! contents (regexp-replace* " \"\\)" contents "\")"))
         ;; Enfin, on ajoute les parenthèses ouvrantes en début de fichier...
         (set! contents (regexp-replace " " contents "((\""))
         ;; ...puis celles en fin de fichier : dans le cas normal...
         (set! contents (regexp-replace "([^\\)])@$" contents "\\1\"))\n"))
         ;; ...et dans le cas où une ligne a été supprimée.
         (set! contents (regexp-replace "\\)@$" contents "))\n"))
         ;; Enfin, on écrit le contenu du fichier modifié vers le fichier de sortie.
         ;; Note : certaines citations pourraient être trop longues pour être envoyées. Cela n'a pas d'importance : elles 
         ;; seront soit découpées lors de l'envoi, soit elle ne seront pas envoyées.
         (display contents output-port)
         ;; Enfin, on ferme le fichier de sortie.
         (close-output-port output-port))))
   ;; On effectue ces opérations pour chaque fichier du dossier de fortunes.
   (directory-list fortunes-directory))
   ;; On affiche enfin un message pour prévenir l'utilisateur.
  (newline)
  (display "La conversion a été effectuée avec succès!")
  (newline))

;; On initialise le chemin du dossier de fortunes.
(define fortunes-path (build-path (current-directory) 'up 'up "fortunes"))
;; On initialise également le chemin du dossier de citations du bot.
(define quotes-path (build-path (current-directory) 'up 'up "citations"))

;; Enfin, on appelle la fonction.
(convert-fortune-fr-files fortunes-path quotes-path)
