;; #############################################################################
;; # ========================================================================= #
;; # Bot IRC Syd - syd.ss                                                      #
;; # Copyright (C) Lozi Jean-Pierre, 2007 - mailto:jean-pierre@lozi.org        #
;; # ========================================================================= #
;; #                                                                           #
;; # This program is free software; you can redistribute it and/or modify it   #
;; # under the terms of the GNU General Public License as published by the     #
;; # Free Software Foundation; either version 2 of the License, or (at your    #
;; # option) any later version.                                                #
;; #                                                                           #
;; # This program is distributed in the hope that it will be useful, but WITH- #
;; # OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or # 
;; # FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for #
;; # for more details.                                                         #
;; #                                                                           #
;; # You should have received a copy of the GNU General Public License along   #
;; # with this program; if not, write to the Free Software Foundation, Inc.,   #
;; # 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.                 #
;; #                                                                           #
;; #############################################################################

;;
;; Note : dans cette application, on parle de "sous-liste" lorsqu'il s'agit d'une liste contenue dans une liste, ce qui n'est
;; pas le sens habituel du terme.
;;

;;
;; Module principal de l'application.
;;
(module syd
  mzscheme
  
  ;;
  ;; On a besoin de cette bibliothèque pour pouvoir inclure des fichiers textuellement.
  ;;
  (require (lib "include.ss"))
  
  ;;
  ;; On a besoin de ce fichier pour manipuler les ports d'entrée et de sortie.
  ;;
  (require (lib "port.ss"))
  
  ;;
  ;; On a besoin de cette bibliothèque pour certaines fonction de manipulation de chaînes de caractères.
  ;;
  (require (lib "string.ss"))
  
  ;;
  ;; On charge le serveur de citations.
  ;;
  (require "syd-quote-server.ss")
  
  ;;
  ;; On charge ce module qui contient des fonctions utilitaires.
  ;;
  (require "syd-utils.ss")
  
  ;;
  ;; On charge le fichier de configuration.
  ;;
  (include (build-path up "syd-configuration.ss"))
  
  ;;
  ;; Port d'entrée (pour l'instant indéfini).
  ;;
  (define input-port '?)
  
  ;;
  ;; Port de sortie (pour l'instant indéfini).
  ;;
  (define output-port '?)
  
  ;;
  ;; On initialise le générateur de nombres aléatoires.
  ;;
  (random-seed (abs (current-milliseconds)))
  
  ;;
  ;; Cette liste contient les états stockés sous la forme de continuations.
  ;;
  (define continuations '())
  
  ;;
  ;; Le serveur de citations doit connaître le canal.
  ;;
  (syd-quote-server-set-channel! channel)
  
  ;;
  ;; Cette variable contient le nickname courant.
  ;;
  (define current-nickname nickname)
  
  ;;
  ;; On initialise le compteur permettant de générer de nouveaux pseudonymes.
  ;;
  (define nickname-number 0)
  
  ;;
  ;; Cette fonction retourne le pseudonyme suivant.
  ;;
  (define (next-nickname)
    (begin0
      ;; On renvoie le pseudonyme suivant...
      (string-append nickname (number->string nickname-number))
      ;; ...et on incrémente le compteur.
      (set! nickname-number (+ nickname-number 1))))
  
  ;;
  ;; Au départ, le pseudonyme alternatif n'a pas été essayé.
  ;;
  (define alternate-nickname-already-tried #f)
  
  ;;
  ;; Boucle principale de gestion des messages.
  ;;
  (define (main-event-loop)
    ;; Notre boucle de gestion des messages est en fait un let.
    (let event-loop
      ;; On crée la liste d'association state, qui représente l'état courant. On l'initialise avec les paramètres par défaut.
      ((state `((mode . normal)
                (mood . good)
                (current-command . none)
                (current-argument . '?)
                (previous-private-message . "")
                (recent-users . ())
                (user-features . ())))
       ;; On initialise la ligne de commande courante avec un read-line
       (line (read-line input-port 'return-linefeed)))
      ;; On affiche les messages reçus, si nécessaire.
      (if display-messages
          (begin
            (display line)
            (newline)))
      ;; On stocke une continuation...
      (let ((new-state (call/cc (lambda (continuation)
                                  ;; ...et on traite le message, en lui passant cette continuation, qui permettra
                                  ;; éventuellement de stocker l'état courant.
                                  (answer-message state line continuation)))))  
        ;; Si la connexion n'est pas fermée...
        (if (not (eof-object? line))
            ;; ...on continue...
            (event-loop
             ;; Si answer-message a renvoyé #f, cela signifie qu'on veut conserver la continuation de la fermeture 
             ;; courante. Sinon, on modifie l'état en continuant à boucler avec le nouvel état.
             (if new-state new-state state)
             ;; Et bien sûr, on passe à la ligne suivante.
             (read-line input-port 'return-linefeed))))))
  
  ;;
  ;; Cette fonction effectue différentes actions selon le message reçu.
  ;;
  (define (answer-message state message continuation)
    ;; On récupère l'emetteur, la commande et les arguments de la ligne de commande.
    (let-values (((sender command args) (syd-utils-parse-message message)))
      ;; On n'utilise pas macro ou de fonction pour appeller automatiquement la bonne commande, car on ne peux pas utiliser
      ;; eval dans un module. Cf. http://list.cs.brown.edu/pipermail/plt-scheme/2007-June/019150.html
      (cond ((string=? command "001") (answer-001-message state sender command args continuation))
            ((string=? command "353") (answer-353-message state sender command args continuation))
            ((string=? command "433") (answer-433-message state sender command args continuation))
            ((string=? command "JOIN") (answer-join-message state sender command args continuation))
            ((string=? command "KICK") (answer-kick-message state sender command args continuation))
            ((string=? command "PART") (answer-part-message state sender command args continuation))
            ((string=? command "PING") (answer-ping-message state sender command args continuation))
            ((string=? command "PRIVMSG") (answer-privmsg-message state sender command args continuation))
            ((string=? command "QUIT") (answer-quit-message state sender command args continuation))
            ;; En tous les cas, on doit renvoyer l'état courant, ou #f pour dire qu'on ne souhaite pas le modifier. Si
            ;; on est tombé sur un type de message qu'on ne traite pas, on ne modifie pas l'état.
            (else #f))))
  
  ;;
  ;; Cette fonction permet de répondre à un message 001.
  ;;
  (define (answer-001-message state server command args continuation)
    ;; On rejoint le canal...
    (syd-utils-send-join-message output-port channel)
    ;; ...sans modifier l'état courant.
    #f)
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message 353, c'est à dire la réponse de la commande NAMES.
  ;;
  (define (answer-353-message state server command args continuation)
    ;; Si cette réponse correspond à une commande !hl...
    (if (eq? (syd-utils-association-list-get state 'current-command) 'hl)
        ;; ...on envoie un message sur le canal (en utilisant toujours le mode normal, pour que le hl fonctionne toujours)...
        (syd-utils-send-private-message output-port 'normal channel
                                        ;; ...contenant...
                                        (string-append
                                         ;; ...une chaîne formatée...
                                         "HL sur " (syd-utils-format-string-from-list
                                                    ;; ...qui contient...
                                                    (syd-utils-randomly-pick-from-list
                                                     ;; ...le bon nombre de personnes choisies aléatoirement...
                                                     (syd-utils-association-list-get state 'current-argument) random
                                                     ;; ...à laquelle on enlève...
                                                     (syd-utils-remove-from-strings
                                                      ;; ...le nom du bot.
                                                      current-nickname
                                                      ;; On enlève les indicateurs de statut devant les noms 
                                                      ;; d'utilisateurs...
                                                      (map syd-utils-drop-status-indicators
                                                           ;; ...qu'on récupère avec un explode...
                                                           (syd-utils-explode 
                                                            ;; ...sur le dernier paramètre, sur lequel on doit faire un
                                                            ;; trim-right...
                                                            (syd-utils-string-trim-right (cadddr args))
                                                            ;; ...le séparateur du explode étant un espace.
                                                            (lambda (v) (eq? v #\space))))))) "!!!")))
    ;; Enfin, on met à jour la commande courante.
    (syd-utils-association-list-set state 'current-command 'none))
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message 443, qui indique que le pseudonyme est déjà choisi.
  ;;
  (define (answer-433-message state sender command args continuation)
    ;; Si on a déjà essayé le pseudonyme alternatif...
    (if alternate-nickname-already-tried
        ;; ...alors, on essaie le prochain nickname...
        (begin (syd-utils-send-nick-message output-port (next-nickname))
               ;; ...et on met à jour la variable current-nickname.
               (set! current-nickname nickname))
        ;; Sinon...
        (begin
          ;; ...on essaie le nickname alternatif...
          (syd-utils-send-nick-message output-port alternate-nickname)
          ;; ...et on se souvient qu'on l'a déjà essayé...
          (set! alternate-nickname-already-tried #t)
          ;; ...et on met à jour la variable current-nickname.
          (set! current-nickname alternate-nickname)))
    ;; On ne modifie pas l'état.
    #f)
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message JOIN.
  ;;
  (define (answer-join-message state sender command args continuation)
    ;; Si le message join ne provient pas de nous...
    (if (not (string=? (syd-utils-nickname sender) current-nickname))
        ;; ...on regarde d'abord si l'utilisateur s'est connecté récemment...
        (if (syd-utils-string-member (syd-utils-username@hostname sender)
                                     (syd-utils-association-list-get state 'recent-users))
            ;; ...si c'est le cas, alors on envoie un message de type "re"...
            (syd-utils-send-random-message-from-list output-port (syd-utils-association-list-get state 'mode)
                                                     (syd-utils-association-list-get state 'mood) channel 
                                                     messages-1/someone-joins-the-channel-again
                                                     (syd-utils-nickname sender))
            ;; ...sinon, on envoie un message de salutations classique.
            (syd-utils-send-random-message-from-list output-port (syd-utils-association-list-get state 'mode)
                                                     (syd-utils-association-list-get state 'mood) channel
                                                     messages-1/someone-joins-the-channel
                                                     (syd-utils-nickname sender))))
    ;; On ne modifie pas l'état courant.
    #f)
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message KICK.
  ;;
  (define (answer-kick-message state sender command args continuation)
    ;; Si on est concerné...
    (if (string=? (cadr args) current-nickname)
        ;; ...on joint le canal à nouveau...
        (begin (syd-utils-send-join-message output-port channel)
               ;; ...et on se plaint!
               (syd-utils-send-random-message-from-list output-port (syd-utils-association-list-get state 'mode)
                                                        (syd-utils-association-list-get state 'mood)
                                                        channel messages-1/i-am-kicked (caddr args)))
        ;; Si c'est quelqu'un d'autre qui s'est fait kicker, on envoie juste un message.
        (syd-utils-send-random-message-from-list output-port (syd-utils-association-list-get state 'mode)
                                                 (syd-utils-association-list-get state 'mood)
                                                 channel messages-1/someone-is-kicked (cadr args)))
    ;; On ne modifie pas l'état courant.
    #f)
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message PART.
  ;;
  (define (answer-part-message state sender command args continuation)
    ;; On utilise la fonction answer-part-or-quit-message.
    (answer-part-or-quit-message state sender command args))
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message PING.
  ;;
  (define (answer-ping-message state sender command args continuation)
    ;; On renvoie juste un message PONG.
    (syd-utils-send-pong-message output-port (car args))
    ;; On ne modifie pas l'état courant.
    
    #f)
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message PRIVMSG.
  ;;
  (define (answer-privmsg-message state sender command args continuation)
    ;; On a besoin d'une return continuation.
    (call/cc 
     (lambda (return)
       ;; On récupère certaines variables.
       (let ((mode (syd-utils-association-list-get state 'mode))
             (mood (syd-utils-association-list-get state 'mood))
             ;; Cette variable nous évite de refaire deux fois les tests d'expressions régulières.
             (current-regexp-result '?))
         
         ;; On traite les fonctions ajoutées pour l'utilisateur.
         (for-each (lambda (element) 
                     ;; On teste l'expression régulière donnée.
                     (let ((regular-expression (regexp-match (car element) (cadr args))))
                       ;; Si on a un match...
                       (if regular-expression
                           ;; ...alors...
                           (with-handlers ((exn:fail?
                                            ;; ...en cas d'erreur...
                                            (lambda (e)
                                              ;; ...on affiche un message d'erreur.
                                              (syd-utils-send-private-message output-port mode channel 
                                                                              "Impossible d'exécuter l'expression.")
                                              state)))
                             ;; On exécute la fonction, et on renvoie le résultat (pour pouvoir modifier l'état).
                             (return ((cdr element) 
                                      regular-expression output-port sender command args state 
                                      ;; On passe quelques variables globales dans une liste d'association.
                                      ;; En fait, il faudrait que celles-ci soient déjà dans une liste
                                      ;; d'association au départ pour que ce soit plus propre. Mais, par manque
                                      ;; de temps, on opte pour cet affreux kludge.
                                      ;;
                                      ;; On pourrait ajouter d'autres variables, sans difficulté. Nous ne le
                                      ;; ferons pas pour garder un code lisible.
                                      `((service-port .  ,service-port)
                                        (channel .  ,channel)
                                        (current-nickname . ,current-nickname)
                                        (first-name . ,first-name)
                                        (last-name . ,last-name)
                                        (connection-message . ,connection-message)
                                        (encoding . ,encoding)
                                        (verbosity-ratio . ,verbosity-ratio)
                                        (default-number-of-nicknames-on-hl . ,default-number-of-nicknames-on-hl)
                                        (default-number-of-recent-users . ,default-number-of-recent-users)
                                        (continuations . ,continuations)
                                        (messages-1/someone-joins-the-channel . ,messages-1/someone-joins-the-channel)
                                        (messages-1/someone-joins-the-channel-again 
                                         . ,messages-1/someone-joins-the-channel-again)
                                        (messages-1/someone-leaves . ,messages-1/someone-leaves)
                                        (messages-1/i-am-kicked . ,messages-1/i-am-kicked)
                                        (messages-1/someone-is-kicked . ,messages-1/someone-is-kicked)
                                        (messages-1/someone-tells-me-o< . ,messages-1/someone-tells-me-o<)
                                        (messages-0/i-leave . ,messages-0/i-leave))))))))
                   ;; On récupère les expressions dans le champ user-features.
                   (syd-utils-association-list-get state 'user-features))
         
         ;; On fait divers tests pour identifier le type de message, en récupérant éventuellement l'état modifié.
         ;; FIXME : il faudrait découper cette fonction en une multitude de fonctions.
         (let ((state                                         
                (cond 
                  ;; ========================================================================================================
                  ;; Si on nous demande d'ajouter une fonction a éxécuter lorsqu'on reçoit une expression régulière...
                  ((begin
                     (let ((regexp-result (regexp-match 
                                           (string-append ".*" current-nickname
                                                          ".*(?i:si quelqu'un dit.*\\\")(.*)(?i:\\\".*ex[eé]cute )(.*)")
                                           (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   
                   ;; ...on l'évalue, s'il y a une erreur, on ne fait rien.
                   (with-handlers ((exn:fail?
                                    (lambda (e)
                                      (syd-utils-send-private-message output-port mode channel 
                                                                      "Impossible d'évaluer l'expression.")
                                      state)))
                     ;; Si tout s'est bien passé, on l'ajoute...
                     (begin0 (syd-utils-association-list-set state 'user-features
                                                     ;; ...l'expression régulière...
                                                     (cons (cons (cadr current-regexp-result)
                                                                 ;; ...et la fonction évaluée.
                                                                 (eval (read-from-string (caddr current-regexp-result))))
                                                           ;; ...à la liste de features.
                                                           (syd-utils-association-list-get state 'user-features)))
                             ;; Enfin, on avertit l'utilisateur.
                             (syd-utils-send-private-message output-port mode channel "Ok !"))))
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande de stocker l'état...
                  ((begin
                     (let ((regexp-result
                            (regexp-match (string-append ".*" current-nickname
                                                         ".*(?i:sauve|enregistre|stocke) (?i:ton [eé]tat sous le nom) (.*)")
                                          (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère le nom souhaité...
                   (let ((name (string->symbol (cadr current-regexp-result))))
                     ;; ...on le stocke, associé à la continuation, dans la liste d'états...
                     (set! continuations (syd-utils-association-list-set continuations name continuation))
                     ;; ...et on avertit l'utilisateur que tout s'est bien passé.
                     (syd-utils-send-private-message output-port mode channel "Ok !"))
                   ;; On ne modifie pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande de restaurer l'état...
                  ((begin
                     (let ((regexp-result (regexp-match
                                           (string-append ".*" current-nickname
                                                          ".*(?i:restaure l'[eé]tat) (.*)") (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère le nom souhaité...
                   (let* ((name (string->symbol (cadr current-regexp-result)))
                          ;; On récupère le couple (name . continuation) correspondant. 
                          (result (assq name continuations)))
                     ;; Si on l'a trouvé...
                     (if result
                         ;; ...on restaure l'état en utilisant la continuation. Comme on renvoie #f, cela veut dire qu'on ne 
                         ;; veut pas modifier l'état. Cela aura pour conséquence de conserver la valeur de l'état *avant* 
                         ;; l'appel de fonction, c'est à dire celui enregistré dans la fermeture de la continuation.
                         ((eval (cdr result)) #f)
                         ;; Sinon, on avertit l'utilisateur...
                         (begin (syd-utils-send-private-message output-port mode channel "Je ne connais pas cet état !")
                                ;; ...et on ne change pas l'état courant.
                                state))))
                  
                  ;; ========================================================================================================
                  ;; Si la même chose a été dite deux fois...
                  ((string=? (cdr (assq 'previous-private-message state)) (cadr args))
                   ;; On la répète.
                   (syd-utils-send-private-message output-port mode channel (cadr args))
                   ;; Enfin, on renvoie l'état tel quel.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande de passer dans un autre mode...
                  ((begin
                     (let ((regexp-result (regexp-match (string-append ".*" current-nickname
                                                                       ".*(?i:passe en mode) (.*)")
                                                        (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère le mode voulu.
                   (let ((mode (string->symbol (cadr current-regexp-result))))
                     ;; Si le mode est connu...
                     (if (memq mode '(normal rainbow hacker majuscules))
                         ;; ...alors on avertit l'utilisateur...
                         (begin (syd-utils-send-private-message output-port mode channel "Ok !")
                                ;; ...et on change le mode.
                                (syd-utils-association-list-set state 'mode mode))
                         ;; Sinon, on avertit l'utilisateur...
                         (begin (syd-utils-send-private-message
                                 output-port (syd-utils-association-list-get state 'mode) channel "Je ne connais pas ce mode !")
                                ;; ...et on ne change pas l'état courant.
                                state))))
                  
                  ;; ========================================================================================================
                  ;; Si on nous pose la fameuse question de Douglas Adams...
                  ((or (regexp-match ".*(?i:answer.*life.*universe.*everything).*" (cadr args))
                       (regexp-match ".*(?i:vie.*univers.*tout le reste).*" (cadr args)))
                   ;; ...on répond "42".
                   (syd-utils-send-private-message output-port mode channel "42")
                   ;; On ne touche pas à l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un nous demande son nom d'hôte...
                  ((or (regexp-match (pregexp (string-append ".*" current-nickname "(?i:.*mon.*h[ôo]te).*")) (cadr args))
                       (regexp-match (pregexp (string-append ".*" current-nickname "(?i:.*mon.*host.*name).*")) (cadr args)))
                   ;; ...on lui répond.
                   (syd-utils-send-private-message output-port mode channel (syd-utils-hostname sender))
                   ;; Et on ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un nous demande son nom d'utilisateur...
                  ((or (regexp-match (pregexp (string-append ".*" current-nickname "(?i:.*mon.*nom.*utilisateur).*"))
                                     (cadr args))
                       (regexp-match (pregexp (string-append ".*" current-nickname "(?i:.*mon.*user.*name).*"))
                                     (cadr args)))
                   ;; ...on lui répond.
                   (syd-utils-send-private-message output-port mode channel (syd-utils-username sender))
                   ;; On ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un nous envoie un "o<"...
                  ((or (regexp-match (string-append ".*" current-nickname ".*[oO0]\\<.*") (cadr args))
                       ;; ...ou un "coin"...
                       (regexp-match (string-append ".*" current-nickname ".*(?i:coin).*") (cadr args)))
                   ;; ...on lui répond.
                   (syd-utils-send-random-message-from-list output-port mode
                                                            (cdr (assq 'mood state)) channel 
                                                            messages-1/someone-tells-me-o<
                                                            (syd-utils-nickname sender))
                   ;; On ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un utilise la commande !hl...
                  ((regexp-match "^!hl.*" (cadr args))
                   ;; ...on regarde s'il y a un paramètre qui suit cette commande...
                   (let ((result (regexp-match "^!hl ([0-9]*)" (cadr args))))
                     ;; On demande la liste de noms du canal.
                     (syd-utils-send-names-message output-port channel)
                     ;; On met à jour la commande courante.
                     (let ((state (syd-utils-association-list-set state 'current-command 'hl)))
                       ;; S'il y a un argument et s'il est valide...
                       (if (and result (number? (read-from-string (cadr result))) (> (read-from-string (cadr result)) 0))
                           ;; ...alors on met à jour l'argument courant.
                           (syd-utils-association-list-set state 'current-argument (read-from-string (cadr result)))
                           ;; Sinon, on choisit l'argument par défaut.
                           (syd-utils-association-list-set state 'current-argument default-number-of-nicknames-on-hl)))))
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande de faire un calcul...
                  ((begin
                     (let ((regexp-result (regexp-match (string-append ".*" current-nickname
                                                                       ".*(?i:calcule|[ée]value) (.*)") (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère ce qu'on nous demande d'évaluer...
                   (let ((expression (cadr current-regexp-result)))
                     ;; ...on crée en nouveau thread...
                     (thread (lambda ()
                               ;; ...en cas d'erreur...
                               (with-handlers ((exn:fail? (lambda (e)
                                                            ;; ...on affiche un message d'erreur.
                                                            (syd-utils-send-private-message output-port mode channel 
                                                                                            "Impossible d'évaluer l'expression.")
                                                            state)))     
                                 ;; Sinon, on calcule et on affiche le résultat.
                                 (syd-utils-send-private-message output-port mode channel
                                                                 (expr->string (eval-string expression)))))))
                   ;; On ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; On répond aux salutations...
                  ((or (regexp-match
                        ;; Si on détecte une salutation après notre nom...
                        (pregexp (string-append ".*" current-nickname
                                                ".*(?:^|[^éèËëàùÜüÏï[:alnum:]])"                                       
                                                "(?i:lu|salut|kikoo+|coucou|bonjour|salut|yop|plop|hello|hey)"
                                                "(?:$|[^éèËëàùÜüÏï[:alnum:]]).*")) (cadr args))
                       (regexp-match
                        ;; ...ou avant notre nom...
                        (pregexp (string-append ".*(?:^|[^éèËëàùÜüÏï[:alnum:]])"
                                                "(?i:lu|salut|kikoo+|coucou|bonjour|salut|yop|plop|hello|hey)"
                                                "(?:$|[^éèËëàùÜüÏï[:alnum:]]).*"
                                                current-nickname ".*")) (cadr args)))
                   ;; ...on par répond un message de salutations.
                   (syd-utils-send-random-message-from-list output-port mode mood channel
                                                            messages-1/someone-joins-the-channel
                                                            ;; ...en spécifiant le pseudonyme de l'utilisateur.
                                                            (syd-utils-nickname sender))
                   ;; On ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; On répond aux "re".
                  ((or (regexp-match
                        ;; Si on détecte un "re" après notre nom...
                        (pregexp (string-append ".*" current-nickname
                                                ".*(?:^|[^éèËëàùÜüÏï[:alnum:]])(?i:re)(?:$|[^éèËëàùÜüÏï[:alnum:]]).*")) 
                        (cadr args))
                       ;; ...ou avant notre nom...
                       (regexp-match
                        (pregexp (string-append ".*(?:^|[^éèËëàùÜüÏï[:alnum:]])(?i:re)(?:$|[^éèËëàùÜüÏï[:alnum:]]).*"
                                                current-nickname ".*")) (cadr args)))
                   ;; ...on par répond un message particulier.
                   (syd-utils-send-random-message-from-list output-port mode mood channel
                                                            messages-1/someone-joins-the-channel-again
                                                            ;; ...en spécifiant le pseudonyme de l'utilisateur.
                                                            (syd-utils-nickname sender))
                   ;; On ne change pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande une citation ou une phrase d'un auteur particulier...
                  ((begin
                     (let ((regexp-result 
                            (regexp-match 
                             (pregexp 
                              (string-append ".*" current-nickname
                                             ".*(?i:citation|quote|message|histoire|texte).*(?i:d[eu]s?) (.*)"))
                             (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère le nom de la source...
                   (let ((source (cadr current-regexp-result)))
                     ;; ...et on demande au serveur de citations de traiter la requête.
                     (write `(display-random-quote-from-source ,mode ,source) syd-quote-server-pipe-output-port)
                     ;; On ne change pas l'état courant.
                     state))
                  
                  ;; ========================================================================================================
                  ;; Si on nous demande une citation ou une phrase d'une source particulière (syntaxe "dis un/une")...
                  ((begin
                     (let ((regexp-result 
                            (regexp-match
                             (pregexp (string-append ".*" current-nickname
                                                     ".*(?i:dis une?) (.*)"))
                             (cadr args))))
                       (set! current-regexp-result regexp-result)
                       regexp-result))
                   ;; ...on récupère le nom de la source...
                   (let ((source (cadr current-regexp-result)))
                     ;; ...et on demande au serveur de citations de traiter la requête.
                     (write `(display-random-quote-from-source ,mode ,source) syd-quote-server-pipe-output-port)
                     ;; On ne change pas l'état courant.
                     state))
                  
                  ;; ========================================================================================================
                  ;; Si on on nous demande d'être de bonne humeur...
                  ((regexp-match (pregexp (string-append ".*" current-nickname
                                                         ".*(?i:sois de bonne humeur).*")) (cadr args))
                   ;; On avertit l'utilisateur...
                   (syd-utils-send-private-message output-port mode channel "D'accord! :)")
                   ;; ...et on le fait.
                   (syd-utils-association-list-set state 'mood 'good))
                  
                  ;; ========================================================================================================
                  ;; Si on on nous demande d'être de mauvaise humeur...
                  ((regexp-match (pregexp (string-append ".*" current-nickname
                                                         ".*(?i:sois de mauvaise humeur).*")) (cadr args))
                   ;; On avertit l'utilisateur...
                   (syd-utils-send-private-message output-port mode channel "Grrr!")
                   ;; ...et on le fait.
                   (syd-utils-association-list-set state 'mood 'bad))
                  
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un dit "di"...
                  ((regexp-match ".*di[^nm].*" (cadr args))
                   ;; ...on envoie le reste de la phrase.
                   (syd-utils-send-private-message output-port mode channel
                                                   (cadr (regexp-match ".*di([^nm].*)" (cadr args))))
                   ;; On ne modifie pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Si quelqu'un dit "cri"...
                  ((regexp-match ".*cri[^nm].*" (cadr args))
                   ;; ...on envoie le reste de la phrase...
                   (syd-utils-send-private-message output-port mode channel
                                                   ;; ...en majuscules...
                                                   (string-append (string-upcase
                                                                   ;; ...avec des "!!!"
                                                                   (cadr (regexp-match ".*cri([^nm].*)" (cadr args)))) " !!!"))
                   ;; On ne modifie pas l'état courant.
                   state)
                  
                  ;; ========================================================================================================
                  ;; Sinon, on demande au serveur de citations de trouver une phrase en rapport avec le message envoyé, avec
                  ;; une certaine probabilité.
                  ((> verbosity-ratio (random))
                   (write `(display-corresponding-quote ,mode ,(cadr args)) syd-quote-server-pipe-output-port)
                   ;; On ne modifie pas l'état courant.
                   state)
                  ;; Dans tout autre cas, on ne modifie pas l'état courant.
                  (else state))))
           ;; Enfin, on met à jour le message reçu précédemment.
           (syd-utils-association-list-set state 'previous-private-message (cadr args)))))))
  
  ;;
  ;; Cette fonction est appellée lors de la réception d'un message QUIT.
  ;;
  (define (answer-quit-message state sender command args continuation)
    ;; On utilise la fonction answer-part-or-quit-message.
    (answer-part-or-quit-message state sender command args))
  
  ;;
  ;; Fonction permettant de traiter en commun les messages PART et QUIT.
  ;;
  (define (answer-part-or-quit-message state sender command args)
    ;; On envoie un message...
    (syd-utils-send-random-message-from-list output-port
                                             (syd-utils-association-list-get state 'mode)
                                             (syd-utils-association-list-get state 'mood) channel messages-1/someone-leaves
                                             ;; ...en précisant celui qui a quitté le canal...
                                             (syd-utils-nickname sender))
    ;; ...puis on met à jour la liste des personnes récemment connectées.
    (syd-utils-association-list-set state 'recent-users (syd-utils-cons-with-limited-size
                                                         default-number-of-recent-users
                                                         (syd-utils-username@hostname sender)
                                                         (syd-utils-association-list-get state 'recent-users))))
  
  ;;
  ;; Boucle principale.
  ;;
  ;; On commence par stocker la continuation pour pouvoir sortir en cas d'erreur.
  ;;
  (call/cc (lambda (return)
             ;; On récupère les erreurs de connexion.
             (with-handlers ((exn:fail:network? 
                              ;; En cas d'erreur de connexion...
                              (lambda (e)
                                ;; ...on affiche un message d'erreur...
                                (display "Une erreur de connexion est survenue.")
                                ;; ...et on quitte l'application.
                                (return 'error)))
                             ;; On récupère les arrêts volontaires de l'application.
                             (exn:break?
                              ;; En cas d'arrêt...
                              (lambda (e)
                                ;; ...on envoie un message QUIT...
                                (syd-utils-send-quit-message output-port
                                                             ;; ...aléatoirement...
                                                             (list-ref messages-0/i-leave (random (length messages-0/i-leave))))
                                ;; ...on ferme les ports...
                                (close-input-port input-port)
                                (close-output-port output-port)
                                ;; ...on affiche un message...
                                (display "L'application s'est arrêtée.")
                                ;; ...et on quitte l'application.
                                (return 'stopped))))
               ;; On se connecte au bon serveur et au bon port.
               (let-values (((raw-input-port raw-output-port)
                             (tcp-connect server-host service-port)))
                 ;; On utilise l'encoding donné.
                 (let ((socket-input-port (reencode-input-port raw-input-port encoding))
                       (socket-output-port (reencode-output-port raw-output-port encoding)))
                   ;; On passe en mode ligne pour le port de sortie.
                   (file-stream-buffer-mode socket-output-port 'line)
                   ;; On enregiste le port d'entrée globalement, pour éviter de toujours le passer en paramètre.
                   (set! input-port socket-input-port)
                   ;; On enregiste le port de sortie globalement, pour éviter de toujours le passer en paramètre.
                   (set! output-port socket-output-port)
                   ;; Le module syd-quote-server a également besoin du port de sortie.
                   (syd-quote-server-set-output-port! socket-output-port)
                   ;; On envoie les commandes initiales : USER...
                   (syd-utils-send-user-message output-port login first-name last-name connection-message)
                   ;; ...et NICK.
                   (syd-utils-send-nick-message output-port nickname)
                   ;; Enfin, on rentre dans la boucle principale de gestion des évènements.
                   (main-event-loop)))))))