;; #############################################################################
;; # ========================================================================= #
;; # Bot IRC Syd - syd-utils.ss                                                #
;; # Copyright (C) Lozi Jean-Pierre, 2007 - mailto:jean-pierre@lozi.org        #
;; # ========================================================================= #
;; #                                                                           #
;; # This program is free software; you can redistribute it and/or modify it   #
;; # under the terms of the GNU General Public License as published by the     #
;; # Free Software Foundation; either version 2 of the License, or (at your    #
;; # option) any later version.                                                #
;; #                                                                           #
;; # This program is distributed in the hope that it will be useful, but WITH- #
;; # OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or # 
;; # FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for #
;; # for more details.                                                         #
;; #                                                                           #
;; # You should have received a copy of the GNU General Public License along   #
;; # with this program; if not, write to the Free Software Foundation, Inc.,   #
;; # 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.                 #
;; #                                                                           #
;; #############################################################################

;;
;; Module utilitaire.
;;
(module syd-utils
  mzscheme
  
  ;;
  ;; On a besoin de cette bibliothèque pour certaines fonctions diverses.
  ;;
  (require (lib "etc.ss"))
  
  ;;
  ;; Formatte une commande et une suite d'arguments comme un message IRC.
  ;;
  ;; Exemple :
  ;;    > (format-command "a" "b" "c" "d" "e")
  ;;    "a b c d :e\r\n"
  ;;
  ;; On pourrait le faire plus simplement avec un reverse, mais on perdrait
  ;; en performances.
  ;;
  (define (format-command-line command . args)
    ;; On crée une fonction format-rec qui fait le travail.
    (letrec ((format-command-line-rec (lambda (result args)
                                        ;; S'il n'y a plus qu'un élément dans la liste...
                                        (if (eq? (cdr args) '())
                                            ;; ...on le concatène au résultat, en ajoutant ":".
                                            (string-append result ":" (car args) "\r\n")
                                            ;; Sinon, on concatène le résultat au premier élément, et on continue avec les
                                            ;; éléments suivants.
                                            (format-command-line-rec (string-append result (car args) " ") (cdr args))))))
      ;; S'il n'y a pas d'arguments...
      (if (null? args)
          ;; ...on renvoie simplement la commande.
          command
          ;; Sinon, on appelle format-rec avec un résultat initial vide.
          (format-command-line-rec "" (cons command args)))))
  
  ;;
  ;; Décompose le message passé en paramètre. Renvoie trois valeurs : l'émetteur (s'il y en a un), la commande, et les
  ;; arguments.
  ;;
  ;; Exemples :
  ;;    > (parse-message ":a b c d :e")
  ;;    "a"
  ;;    "b"
  ;;    ("c" "d" "e")
  ;;
  ;;    > (parse-message "a b c d :e")
  ;;    undef
  ;;    "a"
  ;;    ("b" "c" "d" "e")
  ;;
  (define (parse-message line)
    ;; Fonction qui fait la plus grosse partie du travail : elle décompose le message en une liste de sous-listes,
    ;; chacune de ces sous-listes étant une chaine décomposée en caractères représentant l'un des éléments du message.
    (letrec ((parse-message-helper 
              ;; On utilise un argument result pour être en récursif terminal.
              (lambda (result char-list)
                ;; Si on est arrivés au bout de la chaine de caractère, on renvoie le résultat.
                (cond ((null? char-list) result)
                      ;; Si on tombe sur un espace, on ajoute une nouvelle sous-liste au résultat.
                      ((eq? (car char-list) #\space) (parse-message-helper (cons '() result) (cdr char-list)))
                      ;; Si on tombre sur le caractère ':', on ajoute tout ce qui reste de la liste de caractères comme
                      ;; sous-liste du résultat. 
                      ((eq? (car char-list) #\:) (cons (cdr char-list) (cdr result)))
                      ;; Sinon, on ajoute la lettre courante en fin de la sous-liste courante (note : on utilise append, ce
                      ;; qui n'est pas très efficace).
                      (else (parse-message-helper (cons (append (car result) (list (car char-list))) (cdr result))
                                                  (cdr char-list)))))))
      ;; On transforme d'abord la chaine passée en paramètre en liste.
      (let* ((char-list (string->list line))
             ;; On veut ensuite savoir si la chaîne contient un emetteur.
             (has-sender (eq? (car char-list) #\:))
             ;; Si la liste contient un emetteur...
             (tmp-result (if has-sender
                             ;; On enlève les deux points initiaux avant de traiter la liste.
                             (parse-message-helper '(()) (cdr char-list))
                             ;; Sinon on traite la liste normalement.
                             (parse-message-helper '(()) char-list)))
             ;; Puis on transforme chacune des sous-listes en chaines de caractères. On utilise reverse, ce qui n'est pas
             ;; très efficace.
             (final-list (map (lambda (char-list) (list->string char-list)) (reverse tmp-result))))
        ;; Enfin, on renvoie les valeurs correspondant à l'émetteur, à la commande et aux arguments. 
        (if has-sender
            ;; S'il y a un emetteur, celui-ci est le premier argument.
            (values (car final-list)
                    (cadr final-list)
                    (cddr final-list))
            ;; Sinon, l'emetteur est indefini.
            (values 'undef
                    (car final-list)
                    (cdr final-list))))))
  
  ;;
  ;; Renvoie pseudonyme, nom d'utilisateur et nom d'hôte à partir d'une chaîne "emetteur" de la forme : "nickname!user@host".
  ;; On aurrait aussi pu utiliser notre fonction explode.
  ;;
  (define (nickname-user-host sender)
    ;; On recherche les positions des caractères ! et @.
    (let-values (((!-position @-position)
                  ;; On boucle, en prenant en paramètre ce qui reste de la chaîne de caractères transformée en liste...
                  (let loop ((character-list (string->list sender))
                             ;; ...la position du caractère ! ou '? si celle-ci n'est pas encore disponible...
                             (!-position '?)
                             ;; ...la position du caractère @ ou '? si celle-ci n'est pas encore disponible...
                             (@-position '?)
                             ;; ...et la position initiale (0 au départ).
                             (current-position 0))
                    ;; Si on en est arrivés à la fin de la liste, on renvoie les positions des caractères ! et @...
                    (cond ((null? character-list) (values !-position @-position))
                          ;; ...si on a trouvé le caractère !, on continue avec le reste de la liste, et on stocke la
                          ;; position de ce caractère...
                          ((eq? (car character-list) #\!) (loop (cdr character-list) current-position @-position
                                                                (+ current-position 1)))
                          ;; ...si on a trouvé le caractère @, on n'a plus rien à rechercher si la chaîne est valide, donc
                          ;; on renvoie les positions des caractères ! et @... 
                          ((eq? (car character-list) #\@) (values !-position current-position))
                          ;; ...sinon, on continue à avancer dans la liste.
                          (else (loop (cdr character-list) !-position @-position (+ current-position 1)))))))
      ;; Si l'une des deux valeurs n'a pas été trouvée...
      (if (or (eq? !-position '?) (eq? @-position '?))
          ;; ...on renvoie une erreur.
          (error "Chaîne invalide."))
      ;; On renvoie...
      (values
       ;; ...le pseudonyme...
       (substring sender 0 !-position)
       ;; ...le nom d'utilisateur...
       (substring sender (+ !-position 1) @-position)
       ;; ...et le nom d'hôte.
       (substring sender (+ @-position 1)))))
  
  ;;
  ;; Renvoie le pseudonyme à partir d'une chaîne "emetteur" de la forme : "nickname!username@hostname".
  ;; On utilise nickname-user-host. On pourrait gagner un peu en performances en ne recherchant que ce dont on a besoin.
  ;;
  (define (nickname sender)
    ;; On récupère le pseudonyme, le nom d'utilisateur et le nom d'hôte...
    (let-values (((nickname username hostname) (nickname-user-host sender)))
      ;; ...mais on ne renvoie que le pseudonyme.
      nickname))
  
  ;;
  ;; Renvoie le nom d'utilisateur à partir d'une chaîne "user" de la forme : "nickname!username@hostname".
  ;;
  (define (username sender)
    ;; On récupère le pseudonyme, le nom d'utilisateur et le nom d'hôte...
    (let-values (((nickname username hostname) (nickname-user-host sender)))
      ;; ...mais on ne renvoie que le nom d'utilisateur.
      username))
  
  ;;
  ;; Renvoie le nom d'hôte à partir d'une chaîne "emetteur" de la forme : "nickname!username@hostname".
  ;;
  (define (hostname sender)
    ;; On récupère le pseudonyme, le nom d'utilisateur et le nom d'hote...
    (let-values (((nickname username hostname) (nickname-user-host sender)))
      ;; ...mais on ne renvoie que le nom d'hôte.
      hostname))
 
  ;;
  ;; Renvoie la chaîne username@hostname à partir d'une chaîne "emetteur" de la forme : "nickname!username@hostname".
  ;;
  (define (username@hostname sender)
    ;; On fait un explode sur le caractère '!', et on renvoie le cdr. On pourrait aller plus vite avec une fonction 
    ;; spécifique.
    (cadr (explode sender (lambda (char) (eq? char #\!)))))
  
  ;;
  ;; Cette fonction choisit aléatoirement n éléments (ou moins, si la liste est plus courte) dans une liste en utilisant la
  ;; fonction random-function.
  ;;
  (define (randomly-pick-from-list n random-function elements)
    ;; Si on nous demande de choisir moins d'éléments que le nombre d'éléments disponibles...
    (if (< (length elements) n)
        ;; ...on renvoie la liste initiale.
        elements
        ;; On boucle, en partant avec un résultat vide...
        (let loop ((result '())
                   ;; ...en prenant la liste d'éléments en paramètre...
                   (elements elements)
                   ;; ...et le nombre de mots restants à choisir.
                   (n n))
          ;; Si on a choisi le bon nombre de mots...
          (if (= n 0)
              ;; ...on renvoie le résultat.
              result
              ;; Sinon, on choisit un mot au hasard...
              (let ((index (random-function (length elements))))
                ;; ...on l'ajoute au résultat...
                  (loop (cons (list-ref elements index) result)
                        ;; ...on l'efface de la liste initiale...
                        (list-delete elements index)
                        ;; ...il nous reste donc un mot de moins à choisir.
                        (- n 1)))))))
  
  ;;
  ;; Cette fonction renvoie une liste dans laquelle on a effacé l'élément se trouvant à l'index k.
  ;;
  (define (list-delete list k)
    ;; Si on doit supprimer le premier élément...
    (if (= k 0)
        ;; ...on renvoie le reste de la liste.
        (cdr list)
        ;; Sinon, on efface l'élément k - 1 du reste de la liste.
        (cons (car list) (list-delete (cdr list) (- k 1)))))
  
  ;;
  ;; Cette fonction renvoie une chaîne de caractères identique à celle passée en paramètre à ceci près que les espaces à
  ;; gauche sont supprimés.
  ;;
  (define (string-trim-left string)
    ;; On transforme la chaîne de caractères en une liste.
    (let loop ((characters (string->list string)))
      ;; Tant qu'on a des espaces...
      (if (eq? #\space (car characters))
          ;; ...on avance.
          (loop (cdr characters))
          ;; Lorsqu'on en arrive au premier caractère qui n'est pas un espace, on renvoie ce qui reste de la liste 
          ;; transformée en chaîne de caractères.
          (list->string characters))))
            
  ;;
  ;; Cette fonction renvoie une chaîne de caractères identique à celle passée en paramètre à ceci près que les espaces à 
  ;; droite sont supprimés.
  ;;
  (define (string-trim-right string)
    ;; On transforme la chaîne de caractères en une liste et on la retourne.
    (let loop ((characters (reverse (string->list string))))
      ;; Tant qu'on a des espaces...
      (if (eq? #\space (car characters))
          ;; ...on avance.
          (loop (cdr characters))
          ;; Lorsqu'on en arrive au premier caractère qui n'est pas un espace, on retourne la liste, on la transforme en
          ;; chaîne de caractères, et on renvoie le résultat.
          (list->string (reverse characters)))))
  
  ;;
  ;; Cette fonction renvoie une chaîne de caractères identique à celle passée en paramètre à ceci près que les espaces à 
  ;; droite et à gauche sont supprimés.
  ;;
  (define (string-trim string)
    ;; On utilise simplement les deux fonctions string-trim-left et string-trim-right.
    (string-trim-right (string-trim-left string)))
  
  ;;
  ;; Renvoie une chaîne de caractère basée sur une liste d'éléments, espacés par des virgule ou par un "et" entre les deux
  ;; derniers éléments.
  ;;
  (define (format-string-from-list strings)
    ;; Si la liste ne contient qu'un seul élément...
    (if (null? (cdr strings))
        ;; ...on le renvoie directement.
        (car strings)
        ;; On commence avec un résultat vide...
        (let loop ((result "")
                   ;; ...et la liste de chaînes de caractères initiale.
                   (strings strings))
          ;; Si on en est à l'avant-dernier élément...
          (if (null? (cddr strings))
              ;; ...on le sépare les deux derniers éléments par un "et".
              (string-append result (car strings) " et " (cadr strings))
              ;; Sinon, on sépare les éléments par une virgule.
              (loop (string-append result (car strings) ", ") (cdr strings))))))
  
  ;;
  ;; Cette fonction enlève les indicateurs de statut '+' et '@' au début d'un pseudo.
  ;;
  (define (drop-status-indicators string)
    ;; On transforme la chaîne de caractères en une liste.
    (let ((characters (string->list string)))
      ;; Si le premier caractère est '@' ou '+'...
      (if (memq (car characters) '(#\@ #\+))
          ;; ...alors on renvoie la chaîne initiale à laquelle on a ôté le premier caractère.
          (list->string (cdr characters))
          ;; Sinon, on renvoie la chaîne de caractères initiale.
          string)))
  
  ;;
  ;; Cette fonction découpe une chaîne en une liste de chaînes, étant donné un test qui définit les séparateurs.
  ;;
  ;; Exemple :
  ;;    > (explode "What is the capital of Assyria?" (lambda (char) (memq char '(#\space #\?)))
  ;;    ("What" "is" "the" "capital" "of" "Assyria" "")
  ;;
  (define (explode string separator?)
    ;; Si la chaîne est vide...
    (if (= (string-length string) 0)
        ;; ...on renvoie la liste vide.
        '()
        ;; Sinon, on commence avec une liste qui contient la liste vide.
        (let ((result '(())))
          ;; Pour chaque élément, si on est tombé sur l'un des séparateurs...
          (for-each (lambda (char) (if (separator? char)
                                       ;; ...on ajoute une nouvelle sous-liste dans le résultat.
                                       (set! result (cons '() result))
                                       ;; Sinon, on ajoute le caractère courant à la sous-liste courante.
                                       (set! result (cons (append (car result) (list char)) (cdr result)))))
                    ;; On prend les éléments dans notre chaîne transformée en liste.
                    (string->list string))
          ;; Puis on renverse la liste créée, et on transforme chaque sous-liste en chaîne.
          (map (lambda (list) (list->string list)) (reverse result)))))

  ;;
  ;; Cette fonction renvoie une liste identique à la liste elements, à ceci près que cette liste aura une longueur maximale
  ;; de size : les éléments supplémentaires sont supprimés.
  ;;
  (define (limit-list-length size elements)
    ;; Si la liste est plus courte que la taille demandée...
    (if (> size (length elements))
        ;; ...on la renvoie sans modification.
        elements
        ;; Sinon, on boucle.
        (let loop ((result '()) (size size) (elements elements))
          ;; S'il n'y a plus d'éléments à sélectionner...
          (if (zero? size)
              ;; ...on renvoie le résultat, inversé (car on l'a construit à l'envers).
              (reverse result)
              ;; Sinon, on ajoute le premier élément au résultat, et on demande une liste plus courte d'un élément à partir
              ;; du cdr de la chaîne.
              (loop (cons (car elements) result) (- size 1) (cdr elements))))))
  
  ;;
  ;; Cette fonction ajoute un élément en début de liste, tout en supprimant un élément en fin de liste si celle-ci dépasse
  ;; la taille demandée.
  ;;
  (define (cons-with-limited-size size element elements)
    ;; On ajoute l'élément et on limite la taille de la liste.
    (limit-list-length size (cons element elements)))
  
  ;;
  ;; Fonction similaire à member, mais qui utiliser string=? au lieu de equals?
  ;;
  (define (string-member string strings)
    ;; Si on a atteint la fin de la liste, on renvoie faux.
    (cond ((null? strings) #f)
          ;; Si on a trouvé la chaîne de caractères, on la renvoie.
          ((string=? (car strings) string) string)
          ;; Sinon, on continue à avancer.
          (else (string-member string (cdr strings)))))
  
  ;;
  ;; Envoie un message JOIN correctement formaté.
  ;;
  (define (send-join-message output-port channel)
    (display (format-command-line "JOIN" channel) output-port))
  
  ;;
  ;; Envoie un message NAMES correctement formaté.
  ;;
  (define (send-names-message output-port channel)
    (display (format-command-line "NAMES" channel) output-port))
  
  ;;
  ;; Envoie un message NICK correctement formaté.
  ;;
  (define (send-nick-message output-port nickname)
    (display (format-command-line "NICK" nickname) output-port))
  
  ;;
  ;; Envoie un message PONG correctement formaté.
  ;;
  (define (send-pong-message output-port parameter)
    (display (format-command-line "PONG" parameter) output-port))
  
  ;;
  ;; Envoie un message PRIVMSG correctement formaté.
  ;;
  (define (send-private-message output-port mode channel parameter)
    ;; On récupère la bonne fonction de formattage selon le mode utilisé.
    (let ((format-function (cond ((eq? mode 'normal) identity)
                                 ((eq? mode 'rainbow) to-rainbow)
                                 ((eq? mode 'hacker) to-1337)
                                 ((eq? mode 'majuscules) string-upcase))))
      ;; On construit le message, en utilisant la bonne fonction.
      (let ((message (format-command-line "PRIVMSG" channel (format-function parameter))))
        ;; Si le message est trop long, on ne fait rien.
        (if (<= (string-length message) 450)
            ;; Sinon on envoie le message sur le port de sortie. On pourrait le découper, mais on préfère éviter le flood.
            (display message output-port)))))
  
  ;;
  ;; Cette fonction transforme un texte normal en un texte coloré avec un effet d'arc-en-ciel.
  ;;
  (define (to-rainbow text)
    ;; L'utilisation de ces codes de couleurs dans cet ordre donne un joli effet d'arc-en-ciel.
    (let ((color-list '("13" "06" "04" "05" "07" "08" "09" "03" "10" "11" "12" "02" "06")))
      ;; On boucle, en prenant comme paramètre le résultat courant...
      (let loop ((result "")
                 ;; ...la liste de caractères restant à considérer...
                 (list (string->list text))
                 ;; ...et le numéro de la couleur courante.
                 (color-id 0))
        ;; Si on en est arrivé à la fin de la liste de caractères...
        (if (null? list)
            ;; ...on renvoie le résultat.
            result
            ;; Sinon, on boucle...
            (loop
             ;; ...en rajoutant à la fin du résultat le code spécial 3, la bonne couleur, et la lettre courante...
             (string-append result (string (integer->char 3)) (list-ref color-list color-id) (string (car list)))
             ;; ...on continue avec reste de la liste...
             (cdr list)
             ;; ...et on passe à la couleur suivante.
             (modulo (+ color-id 1) 13))))))
  
  ;;
  ;; Cette fonction transforme un texte normal en un texte 1337.
  ;;
  (define (to-1337 text)
    ;; On crée une liste d'associations pour associer à chaque lettre sa traduction 1337. On garde un niveau relativement
    ;; compréhensible. Pour privilégier certains caractères plus courants, on les ajoute plusieurs fois à une sous-liste.
    (let ((association-list '((a . ("a" "A" "@" "/-\\" "4" "4"))
                              (b . ("b" "B" "8" "6" "|3"))
                              (c . ("c" "C" "[" "<" "("))
                              (d . ("d" "D" "[)" "I>" "|>" "|)"))
                              (e . ("e" "E" "ë" "3" "3" "3" "3"))
                              (f . ("f" "F" "|=" "|#" "ph" "/="))
                              (g . ("g" "G" "6" "6"))
                              (h . ("h" "H" "/-/" "[-]" "]-[" ")-(" "(-)" "|~|" "|-|" "]~[" "}{" "}-{"))
                              (i . ("i" "I" "!" "|" "]" "1" "1" "1"))
                              (j . ("j" "J" "_|" "_/"))
                              (k . ("k" "K" "|<" "|{"))
                              (l . ("l" "L" "|_" "|" "1" "1"))
                              (m . ("m" "M" "|v|" "]V[" "{V}" "|\\/|" "/\\/\\" "(V)" "(\\/)" "/|\\" "^^" "/|/|" "/^^\\"))
                              (n . ("n" "N" "|\\|" "/\\/" "[\\]" "<\\>" "{\\}" "/V"))
                              (o . ("o" "O" "()" "[]" "0" "0"))
                              (p . ("p" "P" "|*" "|º" "|>" "|°"))
                              (q . ("q" "Q" "()_" "O_"))
                              (r . ("r" "R" "|?" "/2" "|2"))
                              (s . ("s" "S" "$" "5" "5"))
                              (t . ("t" "T" "7" "7"))
                              (u . ("u" "U" "(_)" "|_|" "L|"))
                              (v . ("v" "V" "\\/"))
                              (w . ("w" "W" "\\/\\/" "vv" "\\^/" "\\V/" "\\X/" "\\|/" "\\_|_/" "\\_:_/"))
                              (x . ("x" "X" "><" "}{" ")("))
                              (y . ("y" "Y" "`/"))
                              (z . ("z" "Z" "%" ">_" "7_" "2" "2")))))
      ;; On concatène les caractères obtenus...
      (apply string-append
             ;; ...par un map qui, pour chaque caractère...
             (map (lambda (char)
                    ;; ...récupère la sous-liste associée au caractère...
                    (let ((sublist
                           ;; ...en le transformant en symbole et en utilisant assq...
                           (assq (string->symbol (string (char-downcase char)))
                                 ;; ...dans la liste d'association.
                                 association-list)))
                      ;; Si on a trouvé une sous-liste correspondante, le caractère est alphabétique...
                      (if sublist
                          ;; ...donc on récupère la liste de symboles qu'on peut utiliser...
                          (let ((corresponding-symbols (cdr sublist)))
                            ;; ...on en choisit un au hasard.
                            (list-ref corresponding-symbols (random (length corresponding-symbols))))
                          ;; Sinon, on laisse le caractèe initial.
                          (string char))))
                  ;; On récupère chaque caractère dans le paramètre text.
                  (string->list text)))))

  ;;
  ;; Envoie un message QUIT correctement formaté.
  ;;
  (define (send-quit-message output-port message)
    (display (format-command-line "QUIT" message) output-port))
  
  ;;
  ;; Envoie un message USER correctement formaté.
  ;;
  (define (send-user-message output-port login first-name last-name connection-message)
    (display (format-command-line "USER" login first-name last-name connection-message) output-port))
  
  ;;
  ;; Expression régulière de la chaîne spéciale "###", dont le but est d'être remplacée
  ;;.
  (define nickname-special-string-regexp (regexp "###"))
  
  ;;
  ;; Cette fonction sélectionne une phrase aléatoire à partir d'une liste, et remplace la chaîne spéciale "###" par le 
  ;; par le paramètre donné (si celui-ci est spécifié).
  ;;
  (define (random-message-from-list mood list . parameter)
    ;; On récupère la liste de citations...
    (let ((quotes (cdr (assq mood list))))
          ;; Si un paramètre est spécifié...
          (if (not (null? parameter))
              ;; ...on renvoie le message formaté, en l'utilisant.
              (regexp-replace nickname-special-string-regexp (list-ref quotes (random (length quotes))) (car parameter))
              ;; Sinon, on ne l'utilise pas.
              (regexp-replace nickname-special-string-regexp (list-ref quotes (random (length quotes))) (car parameter)))))
  
  ;;
  ;; Cette fonction envoie un message privé obtenu à l'aide de random-message-from list sur le canal donné.
  ;;
  (define (send-random-message-from-list output-port mode mood channel list . parameter)
    ;; Si un paramètre est spécifié...
    (if (not (null? parameter))
        ;; ...on l'utilise.
        (send-private-message output-port mode channel (random-message-from-list mood list (car parameter)))
        ;; Sinon, non.
        (send-private-message output-port mode channel (random-message-from-list mood list))))

  ;;
  ;; Cette fonction renvoie une nouvelle liste d'associations dont la propriété donnée a été modifiée, ou ajoute la propriété
  ;; si celle-ci n'a pas été trouvée.
  ;;
  (define (association-list-set association-list property value)
    ;; On boucle avec un let, en ajoutant un résultat pour être en récursif terminal.
    (let loop ((result '()) (association-list association-list) (property property) (value value))
      ;; Si la liste d'association est vide, alors on ajoute directement la propriété.
      (cond ((null? association-list) (cons (cons property value) result)) 
            ;; Si on a trouvé la propriété, on la modifie et on renvoie le résultat.
            ((eq? property (caar association-list)) (append (cons (cons property value) result) (cdr association-list)))
            ;; Sinon, on continue à avancer.
            (else (loop (cons (car association-list) result) (cdr association-list) property value)))))

  ;;
  ;; Cette fonction renvoie la propriété donnée dans une liste d'associations.
  ;;
  (define (association-list-get association-list property)
    ;; On utilise juste assq.
    (cdr (assq property association-list)))
  
  ;;
  ;; Cette fonction renvoie la liste de chaînes de caractèrs à laquelle on a enlevé la première occurence de la chaîne 
  ;; string. Elle ne conserve pas l'ordre. Si la chaîne n'est pas trouvée, la liste initiale est renvoyée.
  ;;
  (define (remove-from-strings string strings)
    ;; On boucle, en ajoutant une variable result, pour être en récursif terminal.
    (let loop ((result '()) (string string) (strings strings))
      ;; Si la liste est vide, on n'a pas trouvé l'élément : on renvoie donc la liste initiale.
      (cond ((null? strings) strings)
            ;; Si on a trouvé la chaîne de caractère, on renvoie toutes les autres.
            ((string=? (car strings) string) (append result (cdr strings)))
            ;; Sinon, on avance dans la liste.
            (else (loop (cons (car strings) result) string (cdr strings))))))
  
  ;; On exporte tout ce qui a été défini.
  (provide (prefix-all-defined syd-utils-)))