;; #############################################################################
;; # ========================================================================= #
;; # Bot IRC Syd - syd-quote-server.ss                                         #
;; # Copyright (C) Lozi Jean-Pierre, 2007 - mailto:jean-pierre@lozi.org        #
;; # ========================================================================= #
;; #                                                                           #
;; # This program is free software; you can redistribute it and/or modify it   #
;; # under the terms of the GNU General Public License as published by the     #
;; # Free Software Foundation; either version 2 of the License, or (at your    #
;; # option) any later version.                                                #
;; #                                                                           #
;; # This program is distributed in the hope that it will be useful, but WITH- #
;; # OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or # 
;; # FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for #
;; # for more details.                                                         #
;; #                                                                           #
;; # You should have received a copy of the GNU General Public License along   #
;; # with this program; if not, write to the Free Software Foundation, Inc.,   #
;; # 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.                 #
;; #                                                                           #
;; #############################################################################

;; On pourrait créer une macro :
;; (define-macro (module-with-parameters module-identifier initial-required-module-name . parameters-and-body)
;; Celle-ci pourrait permettre d'ajouter des paramètres à un module, et de créer automatiquement les accesseurs. Je l'ai
;; supprimée afin de ne mettre aucune déclaration hors des modules, pour que le code puisse être compilé.

;;
;; Module du serveur de citations.
;;
;; On utilise un thread séparé pour le serveur de citations, pour éviter les ralentissements dans la boucle de traitement 
;; des messages du bot. Je n'ai constaté aucun ralentissement lorsque je n'ai pas utilisé de serveur, mais il est plus 
;; propre de prévoir ce cas (à cause des recherches dans de longues listes de citations).
;;
(module syd-quote-server
  mzscheme
  
  ;;
  ;; On a besoin de cette bibliothèque pour pouvoir inclure des fichiers textuellement.
  ;;
  (require (lib "include.ss"))
  
  ;;
  ;; On a besoin de cette bibliothèque pour certaines fonction de manipulation de listes.
  ;;
  (require (lib "list.ss"))
  
  ;;
  ;; On a besoin de cette bibliothèque pour certaines fonction de manipulation de chaînes de caractères.
  ;;
  (require (lib "string.ss"))
  
  ;;
  ;; On a besoin de cette bibliothèque pour certaines fonction de manipulation de ports d'entrée/sortie.
  ;;
  (require (lib "port.ss"))
  
  ;;
  ;; On a besoin de ce module qui contient des fonctions utilitaires.
  ;;
  (require "syd-utils.ss")
  
  ;;
  ;; On charge le fichier de configuration.
  ;;
  (include (build-path up "syd-quote-server-configuration.ss"))
  
  ;;
  ;; On définit le chemin du répertoire contenant les citations.
  ;;
  (define quotes-path (build-path (current-directory) 'up "citations"))
  
  ;;
  ;; On définit le pipe qui nous permet de discuter avec le reste de l'application.
  ;;
  (define-values (pipe-input-port pipe-output-port) (make-pipe #f 'syd-quote-server-in 'syd-quote-server-out))
  
  ;;
  ;; On a besoin d'avoir accès au port de sortie (de la socket). Pour l'instant, il est indéfini.
  ;;
  (define output-port '?)
  
  ;;
  ;; Cette fonction permet de définir le port de sortie de la socket.
  ;;
  (define (set-output-port! new-output-port)
    ;; On redéfinit juste la variable output-port.
    (set! output-port new-output-port))
  
  ;;
  ;; On a également besoin d'avoir le nom du canal pour pouvoir envoyer des messages. Pour l'instant, le canal est indéfini.
  ;;
  (define channel '?)
  
  ;;
  ;; Cette fonction permet de définir le canal sur lequel envoyer les messages.
  ;;
  (define (set-channel! new-channel)
    ;; On redéfinit juste la variable channel.
    (set! channel new-channel))
  
  ;;
  ;; Initialisation de la liste de citations.
  ;;
  (define quotes 
    ;; On commence avec une liste vide.
    (let ((quotes '()))
      ;; Pour chaque fichier de citations...
      (for-each (lambda (file)
                  ;; On l'ouvre, en utilisant le charset ISO-8859-15.
                  (let ((input-port (reencode-input-port (open-input-file (build-path quotes-path file)) "ISO-8859-15")))
                    ;; On lit la liste de citations contenue dans le fichier.
                    (let ((quote-list (read input-port)))
                      ;; On ajoute la liste, précédée du nom du fichier, à la liste globale de citations.
                      (set! quotes (cons (cons file quote-list) quotes)))))
                ;; Et ce, pour chaque fichier du répertoire "citations".
                (directory-list quotes-path))
      ;; Enfin, on renvoie le résultat.
      quotes))
  
  ;;
  ;; Boucle de traitement des messages.
  ;;
  ;; Dans un nouveau thread...
  (thread (lambda ()
            ;; ...on lit un message...
            (let loop ((pipe-message (read pipe-input-port)))
              ;; ...si le pipe n'a pas été fermé...
              (if (not (eof-object? pipe-message))
                  ;; Si le message est un display-corresponding-quote...
                  (cond ((eq? (car pipe-message) 'display-corresponding-quote)
                         ;; ...alors on recherche et on affiche la citation correspondante.
                         (display-corresponding-quote (cadr pipe-message) (caddr pipe-message)))
                        ;; Si le message est un display-random-quote-from-source...
                        ((eq? (car pipe-message) 'display-random-quote-from-source)
                         ;; ...cette fois encore on recherche et on affiche la citation correspondante.
                         (display-random-quote-from-source (cadr pipe-message) (caddr pipe-message)))))
              ;; On continue avec le message suivant.
              (loop (read pipe-input-port)))))
  
  ;;
  ;; Fonction qui affiche une citation correspondant à un message.
  ;;
  (define (display-corresponding-quote mode irc-message)
    ;; On utilise une continuation pour sortir de la fonction, pour éviter d'avoir trop de niveaux d'imbrication.
    (call/cc (lambda (return)
               ;; On fait exploser le message pour le transformer en une liste de mots. On la découpe selon la ponctuation...
               (let* ((irc-message-exploded (syd-utils-explode irc-message (lambda (char) (or (char-punctuation? char) 
                                                                                              ;; et les espaces blancs.
                                                                                              (char-whitespace? char)))))
                      ;; On filtre le message : on ne s'intéresse qu'aux mots de plus de ignored-word-size-limit caractères. 
                      (irc-message-filtered (filter (lambda (x)
                                                      ;; Si les mots sont trop longs, on les enlève de la liste.
                                                      (> (string-length x) ignored-word-size-limit)) irc-message-exploded)))
                 ;; S'il ne reste aucun mot...
                 (if (= (length irc-message-filtered) 0)
                     ;; ...on ne trouvera aucune citation de toutes façons. La valeur de retour n'a aucune importance.
                     (return (void)))
                 ;; On trie les mots selon leur taille : on définit une fonction string-longer-than?...
                 (let* ((irc-message-sorted (let ((string-longer-than?
                                                   ;; ...qui compare deux mots...
                                                   (lambda (x y)
                                                     ;; ...en voyant lequel est le plus long des deux. Notons qu'on aurait
                                                     ;; pu définir cette fonction globalement, pour éviter sa redéfinition,
                                                     ;; mais il est plus lisible de la déclarer ici.
                                                     (> (string-length x) (string-length y)))))
                                              ;; Puis on trie la liste de mots du plus long au plus court.
                                              (sort irc-message-filtered string-longer-than?)))
                        ;; On tire aléatoirement le nombre de mots à considérer.
                        (number-of-considered-words (random-with-range minimum-number-of-considered-words
                                                                       maximum-number-of-considered-words))
                        ;; On récupère une liste de mots à considérer. On choisit aléatoirement le nombre de mots à
                        ;; considérer. Les plus longs ont plus de chance d'être choisis.
                        (words (select-randomly number-of-considered-words irc-message-sorted)))
                   ;; Si l'utilisateur le désire...
                   (if (and give-up-if-less-words-than-minimum-number-of-considered-words
                            ;; ...et s'il y a moins de mots disponibles que le nombre minimum de mots voulus... 
                            (< (length words) minimum-number-of-considered-words))
                       ;; ...on arrête.
                       (return (void)))
                   ;; Si l'utilisateur le désire...
                   (if (and give-up-if-less-words-than-minimum-number-of-considered-words
                            ;; ...et s'il y a moins de mots disponibles que le nombre de mots voulus...
                            (< (length words) number-of-considered-words))
                       ;; ...on arrête.
                       (return (void)))
                   ;; On transforme la liste de mots obtenue en une liste d'expressions régulières.
                   (let ((regular-expressions (map (lambda (word)
                                                     ;; On ne veut trouver que les mots complets avec ces expressions 
                                                     ;; régulières. Mots au sens large : tous les caractères alphanumériques
                                                     ;; sont inclus. Le pluriel est accepté, d'une manière très basique.
                                                     (pregexp (string-append "(?:^|[^éèËëàùÜüÏï[:alnum:]])"
                                                                             "(?i:" word ")"
                                                                             "(?:[sx]?)(?:$|[^éèËëàùÜüÏï[:alnum:]])"))) 
                                                   ;; Le map s'applique sur la liste de mots.
                                                   words)))
                     ;; On veut récupérer la liste de citations correspondant aux mots sélectionnés. On part d'une liste
                     ;; vide. 
                     (let ((quotes-found '())
                           ;; Si toutes les expressions régulières doivent être présentes dans la phrase...
                           (assert-function (if match-all-words
                                                ;; ...on doit utiliser assert-all...
                                                assert-all
                                                ;; ...sinon, on doit utiliser assert-some.
                                                assert-some)))
                       ;; Pour chaque élément de la sous-liste de citations...
                       (for-each (lambda (sublist)
                                   ;; ...on récupère les citations de la sous-liste...
                                   (let ((quote-sublist (cdr sublist)))
                                     ;; Pour chacune des citations...
                                     (for-each (lambda (quote)
                                                 ;; ...on récupère son contenu...
                                                 (let ((contents (car quote))
                                                       ;; ...on lui assigne un score de zéro au départ..
                                                       (score 0))
                                                   ;; ...on lui applique, avec la bonne fonciton...
                                                   (if (assert-function
                                                        ;; ...toutes les expressions régulières...
                                                        (lambda (regular-expression)
                                                          ;; ...et on récupère le nombre d'occurences...
                                                          (let ((number-of-matches
                                                                 ;; ...pour chaque expression régulière.
                                                                 (length (regexp-match* regular-expression contents))))
                                                            ;; On met à jour le score de cette citation. Au plus on trouve
                                                            ;; d'occurences de l'expression régulière, au plus le score de 
                                                            ;; la citation est élevé.
                                                            (set! score (+ score number-of-matches))
                                                            ;; On s'assure que chaque expression régulière donne des
                                                            ;; résultats pour la citation courante.
                                                            (> number-of-matches 0)))
                                                        ;; Comme nous l'avons dit, cette recherche s'applique dans la liste 
                                                        ;; des expressions régulières.
                                                        regular-expressions)
                                                       ;; Si le test renvoie vrai, on ajoute la citation à la liste des 
                                                       ;; citations trouvées.
                                                       (set! quotes-found (cons (cons score contents) quotes-found)))))
                                               ;; On effectue ces opérations sur la sous-liste de citations.
                                               quote-sublist)))
                                 ;; On effectue ces opérations sur la liste de citations.
                                 quotes)
                       ;; Si on ne trouve aucune citation...
                       (if (= (length quotes-found) 0)
                           ;; ...on abandonne.
                           (return 'ignored))
                       ;; On définit une fonction permettant de comparer le score des citations.
                       (let* ((better-score-than? (lambda (x y)
                                                    ;; On compare leur premier élément.
                                                    (> (car x) (car y))))
                              ;; On trie la liste de citations d'après leur score, du plus élevé au plus faible.
                              (sorted-quote-list (sort quotes-found better-score-than?)))
                         ;; On envoie une citation sur le canal...
                         (syd-utils-send-private-message output-port mode channel 
                          ;; ...choisie parmi les résultats...
                          (cdr (list-ref sorted-quote-list
                                         ;; ...aléatoirement avec une plus grande probabilité d'avoir des citations ayant un
                                         ;; bon score.
                                         (linear-random (length sorted-quote-list)))))))))))))
  
  ;;
  ;; Fonction qui affiche une citation correspondant à un auteur.
  ;;
  (define (display-random-quote-from-source mode source)
    ;; Au départ, la liste des citations trouvées est vide.
    (let ((quotes-found '())
          ;; On crée une expression régulière correspondant à la source demandée.
          (regular-expression (regexp (string-append "(?i:" source ")"))))
      ;; Pour chaque sous-liste de citations...
      (for-each (lambda (sublist)
                  ;; ...on récupère les citations sans l'en-tête...
                  (let ((quote-sublist (cdr sublist)))
                    ;; Pour chaque citation...
                    (for-each (lambda (quote)
                                ;; ...s'il y a une source...
                                (let ((source (if (not (null? (cdr quote)))
                                                  ;; ...on la récupère.
                                                  (cadr quote)
                                                  ;; Sinon, la source est vide.
                                                  "")))
                                  ;; On essaie de faire un match sur la source de la citation.
                                  (if (regexp-match regular-expression source)
                                      ;; En cas de match, on met à jour la liste de citations.
                                      (set! quotes-found (cons (car quote) quotes-found)))))
                              ;; On effectue ces opérations pour chaque élément de la liste de citations.
                              quote-sublist)))
                ;; On récupère les sous-listes à partir de la liste quotes.
                quotes)
      ;; Si on a trouvé des citations...
      (if (not (= (length quotes-found) 0))
          ;; ...on envoie un message sur le canal...
          (syd-utils-send-private-message output-port mode channel
                                          ;; ...contenant l'une des citations, au hasard.
                                          (list-ref quotes-found (random (length quotes-found)))))))
    
               
  
  ;;
  ;; Cette fonction renvoie un entier aléatoire entre x et y compris.
  ;;
  (define (random-with-range x y)
    ;; On utilise la fonction random de scheme.
    (+ x (random (+ (- y x) 1))))
  
  ;;
  ;; Cette fontion renvoie #t si la condition est vérifiée pour chaque élément de la liste.
  ;;
  (define (assert-all condition list)
    ;; On utilise une continuation pour pouvoir sortir.
    (call/cc (lambda (return)
               ;; Pour chaque élément...
               (for-each (lambda (elt)
                           ;; ...si la condition n'est pas vérifiée...
                           (if (not (condition elt))
                               ;; ...on renvoie faux.
                               (return #f)))
                         ;; Les éléments sont pris dans la liste.
                         list)
               ;; Sinon, on renvoie vrai.
               #t)))
  
  ;;
  ;; Cette fontion renvoie #t si la condition est vérifiée pour au moins un élément de la liste.
  ;;
  (define (assert-some condition list)
    ;; On utilise une continuation pour pouvoir sortir.
    (call/cc (lambda (return)
               ;; Pour chaque élément...
               (for-each (lambda (elt)
                           ;; ...si la condition estvérifiée...
                           (if (condition elt)
                               ;; ...on renvoie faux.
                               (return #t)))
                         ;; Les éléments sont pris dans la liste.
                         list)
               ;; Sinon, on renvoie faux.
               #f)))
  
  ;;
  ;; Cette fonction renvoie une liste de number-of-words mots sélectionnés dans la liste de mots donnée, les mots les plus
  ;; longs ayant plus de chance d'être tirés au sort (car ils sont en général plus significatifs). Moins de mots que le
  ;; nombre de mots demandés peuvent être retournés si on ne peutp as faire autrement.
  ;;
  (define (select-randomly number-of-words words)
    ;; On trie les mots selon leur taille : on définit une fonction string-longer-than?...
    (let* ((sorted-list (let ((string-longer-than?
                               ;; ...qui compare deux mots...
                               (lambda (x y)
                                 ;; ...en voyant lequel est le plus long des deux. Notons qu'on aurait pu définir cette
                                 ;; fonction globalement, pour éviter sa redéfinition, mais il est plus lisible de la
                                 ;; déclarer ici.
                                 (> (string-length x) (string-length y)))))
                          ;; Puis on trie la liste de mots du plus long au plus court.
                          (sort words string-longer-than?))))
      ;; On sélectionne au maximum number-of-words mots avec la fonction linear-random sur la liste triée pour avoir plus de
      ;; chances de choisir les mots les plus longs.
      (syd-utils-randomly-pick-from-list number-of-words linear-random sorted-list)))
  
  ;;
  ;; Cette fonction renvoie un nombre aléatoire ayant plus de chances d'être petit que d'être grands entre 0 et k.
  ;;
  (define (linear-random k)
    ;; On prend un entier entre 0 et 1...
    (let ((x (random)))
      ;; ...et on utilise une fonction carré pour privliégier les nombres les plus faibles.
      (inexact->exact (round (* (* x x) (- k 1))))))
  
  ;; On fournit les fonctions de ce modules à l'extérieur.
  (provide (prefix-all-defined syd-quote-server-)))