/*****************************************************************************/
/* ========================================================================= */
/*  tools.c                                                                  */
/*  The tools module contains miscellaneous functions.                       */
/* ========================================================================= */
/*  GenesX - A 3D fractal mountains generator                                */
/*  Copyright (C) Jean-Pierre Lozi, 2005                                     */
/* ========================================================================= */
/*****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*****************************************************************************/
/* Standard headers                                                          */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/*****************************************************************************/
/* Local headers                                                             */
/*****************************************************************************/
#include "tools.h"

/*****************************************************************************/
/* Functions                                                                 */
/*****************************************************************************/
/*
 * Allocates a block of memory of the given size.
 */
void * allocate( int size ) {
    void * result;

    /* We just call the malloc function ... */
    if( ( result = malloc( size ) ) == NULL ) {
        /* ...and check that there was no problem. */
        fprintf( stderr, "Unable to allocate %d bytes of memory.", size );
        exit( EXIT_FAILURE );
    }

    return result;
}

/*
 * Returns a random number with a centered normal distribution with the given
 * variance.
 */
float gauss_random( float variance ) {
    float x1, x2, w, y1;
    static float y2;
    static int use_last = 0;

    /*
     * We use a Box-Muller transformation, for its efficiency.
     * See : http://www.taygeta.com/random/gaussian.html
     */
    if( use_last ) {
        y1 = y2;
        use_last = 0;
    } else {
        do {
            x1 = 2.0 * ( ( rand() * 1.0 ) / RAND_MAX ) - 1.0;
            x2 = 2.0 * ( ( rand() * 1.0 ) / RAND_MAX ) - 1.0;
            w = x1 * x1 + x2 * x2;
        } while ( w >= 1.0 );

        w = sqrt( ( -2.0 * log( w ) ) / w );
        y1 = x1 * w;
        y2 = x2 * w;
        use_last = 1;
    }

    /*
     * Now y1 follows a centered Gaussian distribution. We just have to multiply
     * it with the given variance.
     */
    return( y1 * variance );
}

/*
* This simple function computes the vectorial product of two vectors.
*/
void
cross_product( Vector u, Vector v, Vector result ) {
    float length;

    /* We just compute the cross product... */
    result[0] = ( u[1] * v[2] - v[1] * u[2] );
    result[1] = -( u[0] * v[2] - v[0] * u[2] );
    result[2] = ( u[0] * v[1] - u[1] * v[0] );

    /* ...and normalize it. */
    length = sqrt( result[0] * result[0] + result[1] * result[1] +
                   result[2] * result[2] );

    result[0] /= length;
    result[1] /= length;
    result[2] /= length;
}

/*
 * This simple function returns the difference between the vector a and the
 * vector b.
 */
void
difference( Vector a, Vector b, Vector result ) {
    /* We just substract the coordinates. */
    result[0] = a[0] - b[0];
    result[1] = a[1] - b[1];
    result[2] = a[2] - b[2];
}
