/*****************************************************************************/
/* ========================================================================= */
/*  rendering.c                                                              */
/*  The rendering module contains the rendering/3D functions.                */
/* ========================================================================= */
/*  GenesX - A 3D fractal mountains generator                                */
/*  Copyright (C) Jean-Pierre Lozi, 2005                                     */
/* ========================================================================= */
/*****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*****************************************************************************/
/* OpenGL headers                                                            */
/*****************************************************************************/
#include <GL/gl.h>
#include <GL/glu.h>

/*****************************************************************************/
/* Standard headers                                                          */
/*****************************************************************************/
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/*****************************************************************************/
/* Local headers                                                             */
/*****************************************************************************/
#include "tools.h"
#include "globals.h"
#include "constants.h"
#include "generation.h"
#include "rendering.h"

/*****************************************************************************/
/* Type definitions                                                          */
/*****************************************************************************/
typedef struct
{
  unsigned long sizeX;
  unsigned long sizeY;
  char *data;
}
Image;

/*****************************************************************************/
/* Symbolic constats                                                         */
/*****************************************************************************/
#define REPEATS_PER_CELL 100

/*****************************************************************************/
/* Global variables                                                          */
/*****************************************************************************/
/* OpenGL constants. */
static GLuint landscape_texture, water_texture;
static Image *landscape_image = NULL, *water_image = NULL;

/*****************************************************************************/
/* Function prototypes                                                       */
/*****************************************************************************/

/*gentoo
 * This function renders the ground.
 */
static void
render_ground( void );

/*
 * This function renders the mountains.
 */
static void
render_mountains( void );

/*
 * This function renders the water.
 */
static void
render_water( void );

/*
 * This function initializes the view.
 */
static void
init_view( void );

/*
 * This function loads the GL textures.
 */
static GLvoid
load_gl_textures( GLvoid );

/*
 * This function loads the file with the given filename into an Image object.
 */
static int
image_load( char *filename, Image *image );

/*****************************************************************************/
/* Functions                                                                 */
/*****************************************************************************/

/*
* This function initializes the parameters of our OpenGL window.
*/
void init_gl( int width, int height )
{
  /* We first load the textures. */
  load_gl_textures();

  /* We enable or disable the textures depending on the user choice. */
  if( !g_textures )
    glDisable( GL_TEXTURE_2D );
  else
    glEnable( GL_TEXTURE_2D );

/*
  glEnable(GL_POLYGON_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_POINT_SMOOTH);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glBindTexture( GL_TEXTURE_2D, landscape_texture );
*/
  /* We enable the color material. */
  glEnable( GL_COLOR_MATERIAL );
  /* Shade model. */
  if( g_smooth_normals )
    glShadeModel( GL_SMOOTH );
  else
    glShadeModel( GL_FLAT );
  /* We enable the lighting. */
  glEnable( GL_LIGHTING );
  /* Really nice perspective calculations. */
  glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );
  /* We set the background. */
  glClearColor( g_sky_color[0], g_sky_color[1], g_sky_color[2], 1.0f );
  /* Now we set the fog, if needed. */
  if( g_fog )
  {
    /* We use a linear fog. */
    glFogi( GL_FOG_MODE, GL_LINEAR );
    /* We set the fog color. */
    glFogfv( GL_FOG_COLOR, g_sky_color );
    /* We set the fog density. */
    glFogf( GL_FOG_DENSITY, g_fog_density );
    /* We set the fog hint value. */
    glHint( GL_FOG_HINT, GL_NICEST );
    /* We set the fog start depth. */
    glFogf( GL_FOG_START, g_fog_start );
    /* We set the fog end depth. */
    glFogf( GL_FOG_END, g_fog_end );
    glEnable( GL_FOG );
  }
  else
    glDisable( GL_FOG );
  /* We enable the depth testing. */
  glEnable( GL_DEPTH_TEST ); //glEnable( GL_CULL_FACE );
  /* We enable the clearing of the depth buffer. */
  glClearDepth( 1.0 );
  /* We choose the type of depth test to do. */
  glDepthFunc( GL_LESS );
  /* We reset the projection matrix. */
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();
  /* We calculate the aspect ration of the window. */
  gluPerspective( g_field_of_view_angle, ( GLfloat )width/( GLfloat )height, 0.1f, g_perspective );
  glMatrixMode( GL_MODELVIEW );
  
  /* Now we load the light. */
  glLightfv( GL_LIGHT1, GL_AMBIENT, g_light_ambient_color );
  glLightfv( GL_LIGHT1, GL_DIFFUSE, g_light_diffuse_color );
  glLightfv( GL_LIGHT1, GL_POSITION, g_light_position );
  glEnable( GL_LIGHT1 );

  /* We enable the lighting and the blending. */
  glEnable( GL_LIGHTING );
  glEnable( GL_BLEND );

}

/*
* This is the main drawing function.
*/
void draw_gl_scene()
{
  /* We clear the color buffer and the depth buffer. */
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  if( g_landscape == NULL )
    return;

  glLoadIdentity();

  /* We initialize the view. */
  init_view();

  if( !g_reflection )
  {
    /* We render the ground... */
    if( g_ground )
      render_ground ();

    /* ...the mountains... */
    render_mountains();

    /* ...and the water. */
    render_water();
  }
  else
  {
    g_water_level = 0;


    glPushMatrix();
  
    render_mountains();
    glScalef( 1.0f, 1.0f, -1.0f );
    glLightfv( GL_LIGHT1, GL_AMBIENT, g_light_ambient_color );
    glLightfv( GL_LIGHT1, GL_DIFFUSE, g_light_diffuse_color );
    glLightfv( GL_LIGHT1, GL_POSITION, g_light_position );
    glEnable( GL_LIGHT1 );
    render_mountains();
    glPopMatrix();

    render_water();
  }
}

/*
 * This function renders the mountains.
 */
static void
render_mountains( void )
{
  register int i, j;
  int next_i, next_j, previous_j;
  int num_divisions_max = g_num_divisions - 1;
  
  /* We set the blending function. */
  glBlendFunc (GL_ALPHA, GL_ZERO);

  if( g_textures )
    glBindTexture( GL_TEXTURE_2D, landscape_texture );

  /* Now we start drawing the polygons. */
  for( i = 0 ; i < num_divisions_max ; i++ )
  {
    j = 0;
    next_i = i + 1;
    next_j = 1;
    
    glBegin( GL_TRIANGLE_STRIP );

    /* We draw the first triangle with its normal. */
    if( g_smooth_normals )
      glNormal3fv( g_landscape[i][j].bottom_right_smooth_normal );
    else
      glNormal3fv( g_landscape[i][j].bottom_right_normal ),

    glColor3fv( g_landscape[i][j].color );
    glTexCoord2f( 0.0f, 0.0f );
    glVertex3fv( g_landscape[i][j].position );

    glColor3fv( g_landscape[i][next_j].color );
    glTexCoord2f( 0.0f, 1.0f );
    glVertex3fv( g_landscape[i][next_j].position );

    glColor3fv( g_landscape[next_i][j].color );
    glTexCoord2f( 1.0f, 0.0f );
    glVertex3fv( g_landscape[next_i][j].position );

    glColor3fv( (GLfloat *)g_landscape[i+1][j+1].color );

    /* Now we draw the second triangle, with its normal. */
    if( g_smooth_normals )
      glNormal3f( g_landscape[next_i][next_j].top_left_smooth_normal[0],
                  g_landscape[next_i][next_j].top_left_smooth_normal[1],
                  g_landscape[next_i][next_j].top_left_smooth_normal[2] );
    else
      glNormal3fv( g_landscape[next_i][next_j].top_left_normal );

    glTexCoord2f( 1.0f, 1.0f );
    glVertex3fv( g_landscape[next_i][next_j].position );
    glColor3fv( (GLfloat *)g_landscape[next_i][next_j].color );


    for( j = 1 ; j < g_num_divisions ; j++ )
    {
      previous_j = j - 1;
      
      if( g_smooth_normals )
        glNormal3fv( g_landscape[i][previous_j].bottom_right_smooth_normal );
      else
        glNormal3fv( g_landscape[i][previous_j].bottom_right_normal );

      /* We draw the first triangle with its normal. */
      glTexCoord2f( j, 0.0f );
      glColor3fv( g_landscape[i][j].color );
      glVertex3fv( g_landscape[i][j].position );

      /* Now we draw the second triangle, with its normal. */
      if( g_smooth_normals )
        glNormal3fv( g_landscape[next_i][j].top_left_smooth_normal );
      else
        glNormal3fv( g_landscape[next_i][j].top_left_normal );

      glColor3fv( g_landscape[next_i][j].color );
      glTexCoord2f( j, 1.0f );
      glVertex3fv( g_landscape[next_i][j].position );
    }
    glEnd();
  }
}

/*
 * This function renders the ground.
 */
static void
render_ground( void )
{
  float x_acc, y_acc;
  float step;
  float cell_size, x_offset, y_offset;

  cell_size = 2 * g_perspective / g_ground_precision;

  if( g_textures )
    glBindTexture( GL_TEXTURE_2D, landscape_texture );
  
  glColor3fv( g_colors_array[0] );

  x_offset = - g_position[0] / cell_size;
  y_offset = - g_position[1] / cell_size;

  glBlendFunc ( GL_ONE, GL_ZERO );

  if( !g_fog )
  {
    glBegin( GL_QUADS );
    glNormal3f( 0, 0, 1.0f );
    glTexCoord2f( x_offset,
                  y_offset );
    glVertex3f( -g_perspective - g_position[0],
                -g_perspective - g_position[1],
                0 );
    glTexCoord2f( x_offset,
                  y_offset + g_ground_precision );
    glVertex3f( -g_perspective - g_position[0],
                g_perspective - g_position[1],
                0 );
    glTexCoord2f( x_offset + g_ground_precision,
                  y_offset + g_ground_precision );
    glVertex3f( g_perspective - g_position[0],
                g_perspective - g_position[1],
                0 );
    glTexCoord2f( x_offset + g_ground_precision,
                  y_offset );
    glVertex3f( g_perspective - g_position[0],
                -g_perspective - g_position[1],
                0 );
    glEnd();
  }
  /* If there is some fog, we need to draw each cell separaterly. */
  else
  {
    glBegin( GL_QUADS );
    step = 2 * g_perspective / ( g_ground_precision / REPEATS_PER_CELL );
    for( x_acc = -g_perspective - g_position[0] ;
         x_acc < g_perspective - g_position[0] ;
         x_acc += step )
    {
      for( y_acc = -g_perspective - g_position[1] ;
           y_acc < g_perspective - g_position[1] ;
           y_acc += step )
      {
        glNormal3f( 0.0f, 0.0f, 1.0f );
        glTexCoord2f( x_offset, 0.0f + y_offset );
        glVertex3f( x_acc,
                    y_acc,
                    0 );
        glTexCoord2f( x_offset, REPEATS_PER_CELL + y_offset );
        glVertex3f( x_acc,
                    y_acc + step,
                    0 );
        glTexCoord2f( REPEATS_PER_CELL + x_offset, REPEATS_PER_CELL + y_offset );
        glVertex3f( x_acc + step,
                    y_acc + step,
                    0 );
        glTexCoord2f( REPEATS_PER_CELL + x_offset, y_offset );
        glVertex3f( x_acc + step,
                    y_acc,
                    0 );
      }
    }
    glEnd();
  }
}

/*
 * This function renders the water.
 */
static void
render_water( void )
{
  float x_acc, y_acc;
  float dec, step, x_offset, y_offset;
  float cell_size;

  cell_size = 2 * g_perspective / g_water_precision;

  /*
   * We set a texture or a color, depending on the current configuration.
   */
  if( g_textures )
  {
    glBindTexture( GL_TEXTURE_2D, water_texture );
    glColor4f( 1.0f, 1.f, 1.f, g_water_opacity);
  }
  else
  {
    glColor4f( .5f, .5f, 1.f, g_water_opacity);
  }
  dec = ( g_water_pos_counter * g_water_step );


  x_offset = dec - g_position[0] / cell_size;
  y_offset = dec - g_position[1] / cell_size;

  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  if( !g_fog )
  {
    glBegin( GL_QUADS );
    glNormal3f( 0,0 ,-1.0f );
    glTexCoord2f( x_offset,  y_offset  );
    glVertex3d( -g_perspective - g_position[0],
                -g_perspective - g_position[1],
                g_water_level );
    glTexCoord2f( x_offset,
                  g_water_precision  + y_offset);
    glVertex3d( -g_perspective - g_position[0],
                g_perspective - g_position[1],
                g_water_level );
    glTexCoord2f( g_water_precision + x_offset ,
                  g_water_precision + y_offset);
    glVertex3d( g_perspective - g_position[0],
                g_perspective - g_position[1],
                g_water_level );
    glTexCoord2f( g_water_precision + x_offset ,
                  y_offset);
    glVertex3f(  g_perspective  - g_position[0],
                 -g_perspective - g_position[1],
                 g_water_level );
    glEnd();
  }
  /* We have to draw each cell separately. */
  else
  {
    glBegin( GL_QUADS );
    step = 2 * g_perspective / ( g_water_precision / REPEATS_PER_CELL );
    for( x_acc = -g_perspective - g_position[0] ;
         x_acc < g_perspective - g_position[0] ;
         x_acc += step )
    {
      for( y_acc = -g_perspective - g_position[1] ;
           y_acc < g_perspective - g_position[1] ;
           y_acc += step )
      {
        glNormal3f( 0,0 ,-1.0f );
        glTexCoord2f( x_offset,
                      + y_offset );
        glVertex3f( x_acc,
                    y_acc,
                    g_water_level );
        glTexCoord2f( x_offset,
                      REPEATS_PER_CELL + y_offset );
        glVertex3f( x_acc,
                    y_acc + step,
                    g_water_level );
        glTexCoord2f( REPEATS_PER_CELL + x_offset,
                      REPEATS_PER_CELL + y_offset );
        glVertex3f( x_acc + step,
                    y_acc + step,
                    g_water_level );
        glTexCoord2f( REPEATS_PER_CELL + x_offset,
                      y_offset );
        glVertex3f( x_acc + step,
                    y_acc,
                    g_water_level );
      }
    }
    glEnd();
  }
}

/*
* This function called when our window is resized.
*/
void
resize_gl_scene( int width, int height )
{
  /* We want to avoid a Divide By Zero error if the window is too small. */
  if( height == 0 )
    height = 1;

  /* Now we reset the current viewport ant the perspective transformation. */
  glViewport( 0, 0, width, height );

  /* We reset the projection matrix. */
  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();
  /* We calculate the aspect ration of the window. */
  gluPerspective( g_field_of_view_angle, ( GLfloat )width/( GLfloat )height, 0.1f, g_perspective );
  glMatrixMode( GL_MODELVIEW );
}

/*
 * This function loads the GL textures.
 */
static GLvoid
load_gl_textures( GLvoid )
{
  if( g_load_textures )
  {
    landscape_texture = 0;
    water_texture = 1;

    if( landscape_image == NULL ) free( landscape_image );
    if( water_image == NULL ) free( water_image );
    
    landscape_image = ( Image * )malloc( sizeof( Image ) );
    water_image = ( Image * )malloc( sizeof( Image ) );
    
    if ( ( landscape_image == NULL ) || ( water_image == NULL ) )
    {
      printf("Error allocating space for image");
      exit(0);
    }

    if ( !image_load( g_landscape_texture_path, landscape_image ) )
      exit( NUM_TEXTURES );

    if ( !image_load( g_water_texture_path, water_image ) )
      exit( NUM_TEXTURES );

    g_load_textures = 0;
  }
  /* Then we create each texture. */

  /* 2D Texture. */
  glBindTexture(GL_TEXTURE_2D, landscape_texture );

  /* We use a linear scaling. */
  if( g_filtered_textures )
  {
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
  }
  else
  {
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  }
  
  glTexImage2D( GL_TEXTURE_2D, 0, 3, landscape_image->sizeX,
                landscape_image->sizeY, 0, GL_RGB, GL_UNSIGNED_BYTE,
                landscape_image->data );
                
  /* 2D Texture. */
  glBindTexture(GL_TEXTURE_2D, water_texture );

  /* We use a linear scaling. */
  if( g_filtered_textures )
  {
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
  }
  else
  {
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  }
  
  glTexImage2D( GL_TEXTURE_2D, 0, 3, water_image->sizeX,
                water_image->sizeY, 0, GL_RGB, GL_UNSIGNED_BYTE,
                water_image->data );
}

/*
 * This function initializes the view.
 */
static void
init_view( void )
{
  Vector u1, u2, u3;

  u1[0] = cos( g_theta * PI_OVER_180 );
  u1[1] = sin( g_theta * PI_OVER_180 );
  u1[2] = 0;

  u2[0] = 0;
  u2[1] = 0;
  u2[2] = -1.0;

  cross_product( u1, u2, u3 );
  glColor4f(1.f,1.f,1.f,1.0f);

  glRotatef( -90, 1.0f, 0.0f, 0.0f );
  glRotatef( -90, 0.0f, 0.0f, 1.0f );

  glRotatef( g_theta, 0.0f, 0.0f, 1.0f );
  glRotatef( g_phi, u3[0], -u3[1], u3[2] );
  glTranslatef( g_position[0], g_position[1] , -g_position[2] );
}

/*
 * This function loads the file with the given filename into an Image object.
 */
static int
image_load( char *filename, Image *image )
{
  FILE *file;
  /* The size of the image. */
  unsigned long size;
  /* A simple counter. */
  unsigned long i;
  /* The number of planes in the image (must be 1). */
  unsigned short int planes;
  /* The number of bits per pixel (must be 24). */
  unsigned short int bpp;
  /* Used to convert BGR to RGB color. */
  char temp;

  /* First, we open the file. */
  if( ( file = fopen(filename, "rb" ) ) == NULL )
  {
    fprintf( stderr, "File Not Found : %s\n" , filename );
    return 0;
  }

  /* We seek through the bmp header, up to the width/height. */
  fseek(file, 18, SEEK_CUR);

  /* We read the width. */
  if( ( i = fread(&image->sizeX, 4, 1, file ) ) != 1 )
  {
    fprintf( stderr, "Error reading width from %s.\n", filename );
    return 0;
  }
  printf( "Width of %s: %lu\n", filename, image->sizeX );

  /* We read the height. */
  if( ( i = fread( &image->sizeY, 4, 1, file ) ) != 1 )
  {
    fprintf( stderr, "Error reading height from %s.\n", filename );
    return 0;
  }
  printf( "Height of %s: %lu\n", filename, image->sizeY );

  /* We calculate the size (assuming 24 bits or 3 bytes per pixel). */
  size = image->sizeX * image->sizeY * 3;

  /* We read the planes */
  if( ( fread( &planes, 2, 1, file ) ) != 1 )
  {
    fprintf( stderr, "Error reading planes from %s.\n", filename );
    return 0;
  }
  if( planes != 1 )
  {
    fprintf( stderr, "Planes from %s is not 1: %u\n", filename, planes );
    return 0;
  }

  /* We read the bpp. */
  if( ( i = fread(&bpp, 2, 1, file ) ) != 1 )
  {
    fprintf( stderr, "Error reading bpp from %s.\n", filename);
    return 0;
  }
  if( bpp != 24 )
  {
    fprintf( stderr, "Bpp from %s is not 24: %u\n", filename, bpp);
    return 0;
  }

  /* We seek past the rest of the bitmap header. */
  fseek( file, 24, SEEK_CUR );

  /* We read the data. */
  image->data = ( char * )malloc( size );
  if( image->data == NULL )
  {
    fprintf( stderr, "Error allocating memory for color-corrected image data.\n" );
    return 0;
  }

  if( ( i = fread( image->data, size, 1, file ) ) != 1 )
  {
    fprintf( stderr, "Error reading image data from %s.\n", filename );
    return 0;
  }

  for( i = 0 ; i < size ; i += 3 )
  { /* We reverse all of the colors. (bgr -> rgb) */
    temp = image->data[i];
    image->data[i] = image->data[i+2];
    image->data[i+2] = temp;
  }

  /* We're done. */
  return 1;
}
