/*****************************************************************************/
/* ========================================================================= */
/*  gui.c                                                                    */
/*  The GUI module contains the user interface related functions.            */
/* ========================================================================= */
/*  GenesX - A 3D fractal mountains generator                                */
/*  Copyright (C) Jean-Pierre Lozi, 2005                                     */
/* ========================================================================= */
/*****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*****************************************************************************/
/* Local headers                                                             */
/*****************************************************************************/
#include "constants.h"
#include "files.h"
#include "generation.h"
#include "gui.h"
#include "globals.h"

/*****************************************************************************/
/* Standard headers                                                          */
/*****************************************************************************/
#include <math.h>
#include <stdlib.h>
#include <string.h>

/*****************************************************************************/
/* Gtk+/GtkGLExt headers                                                     */
/*****************************************************************************/
#include <gtk/gtk.h>
#include <gtk/gtkgl.h>
#include <gdk/gdkkeysyms.h>

/*****************************************************************************/
/* OpenGL headers                                                            */
/*****************************************************************************/
#include <GL/gl.h>
#include <GL/glu.h>

/*****************************************************************************/
/* Static functions                                                          */
/*****************************************************************************/

/*
 * This function creates the menus for our application,
 * and returns the menu bar.
 */
static GtkWidget *
create_menus( GtkWidget * main_window );

/*
 * This function is called each time an expose_event is received.
 */
static int
expose_event ( GtkWidget      *widget,
               GdkEventExpose *event,
               gpointer        data );

/*
 * This function is called when the window is resized.
 */
static gboolean
reshape ( GtkWidget         *widget,
          GdkEventConfigure *event,
          gpointer           data );

/*
 * This function adds an idle state to the given object.
 */
static void
idle_add( GtkWidget *widget );

/*
 * This function removes the given object's idle state.
 */
static void
idle_remove( GtkWidget *widget );

/*
 * This function is called each time a map event is received.
 */
static gboolean
map( GtkWidget   *widget,
     GdkEventAny *event,
     gpointer     data );

/*
 * This function is called each time an unmap event is received.
 */
static gboolean
unmap( GtkWidget   *widget,
       GdkEventAny *event,
       gpointer     data );

/*
 * This function is called each time a key is pressed.
 */
static gboolean
on_key_press_event( GtkWidget *widget, GdkEventKey *event, gpointer data);

/*
 * This function displays the "New Landscape..." dialog.
 */
static void
on_help_about( GtkWidget *widget, GdkEventKey *event, gpointer data );

/*
 * This function displays the "New Landscape..." dialog.
 */
int
display_new_landscape_dialog( void );

/*
* This function displays the "Properties..." dialog.
*/
int
display_properties_dialog( void );

/*
 * This function is called when the user chooses the "Properties..." item from
 * the "Parameters" menu.
 */
static void
on_parameters_properties( GtkBox *widget, GdkEventKey *event, gpointer data );

/*
 * This function is called when the user chooses the "New Landscape..." item
 * from the "File" menu.
 */
static void
on_file_new( GtkWidget *widget, GdkEventKey *event, gpointer data );

/*
 * This function creates a line with a spin button in the given container,
 * and returns the spin button.
 */
static GtkWidget *
create_spin_button_line( GtkBox *parent, char *label_text,
                         float initial_value, float min_value,
                         float max_value, float step, int digits );
/*
 * This function creates a line with a check button in the given container,
 * and returns the check button.
 */
static GtkWidget *
create_check_button_line( GtkBox *parent, char *label_text,
                          float initial_value );

/*
 * This function creates a line with a color button,
 * and returns the latter.
 */
static GtkWidget *
create_color_button_line( GtkBox *parent, char *label_text,
                          float red, float green, float blue,
                          float alpha, int use_alpha );

/*
 * This function creates a line with a file selection widget
 */
static GtkWidget *
create_file_chooser_button_line( GtkBox *parent, char *initial_path, char *label_text );

/*
 * This function creates a simple frame.
 */
static GtkWidget *
create_frame( GtkWidget * parent, char *caption );

/*
 * This is called when the user chooses "Save" from the "File" menu.
 */
static void
on_file_save( GtkWidget *widget, GdkEventKey *event, gpointer data );

/*
 * This is called when the user chooses "Save as..." from the "File" menu.
 */
static void
on_file_save_as( GtkWidget *widget, GdkEventKey *event, gpointer data );

/*
 * This is called when the user chooses "Open..." from the "File" menu.
 */
static void
on_file_open( GtkWidget *widget, GdkEventKey *event, gpointer data );

/*****************************************************************************/
/* Static variables                                                          */
/*****************************************************************************/
static guint idle_id = 0;
static GTimer *timer;

/* The list of the menu items. */
static GtkItemFactoryEntry menu_items[] = {
            {"/_File",                      NULL,         NULL, 0, "<Branch>" },
            {"/_File/_New Landscape...",    "<control>N", on_file_new, 0, NULL },
            {"/_File/_Open...",             "<control>O", on_file_open, 0, NULL },
            { "/File/File_Separator_2",     NULL,         NULL, 0, "<Separator>" },
            { "/File/_Save",                "<control>S", on_file_save, 0, NULL },
            { "/File/Save _As...",          NULL,         on_file_save_as, 0, NULL },
            { "/File/File_Separator_2",     NULL,         NULL, 0, "<Separator>" },
            { "/File/Quit",                 "<control>Q", gtk_main_quit, 0, NULL },
            { "/_Parameters",               NULL,         NULL, 0, "<Branch>" },
            { "/Parameters/Properties...",  "<control>P", on_parameters_properties, 0, NULL },
            { "/_Help",                     NULL,         NULL, 0, "<Branch>" },
            { "/_Help/_About",              "<control>H", on_help_about, 0, NULL },
        };

static GtkWidget *main_window, *drawing_area;

static gboolean is_sync = TRUE;

/*****************************************************************************/
/* Functions                                                                 */
/*****************************************************************************/

/*
 * This function creates the main window, with all its components.
 */
void
create_main_window( float width, float height ) {

    /* The window's widgets. */
    GtkWidget *menu_bar, *vertical_box;
    /* The GL configuration. */
    GdkGLConfig *gl_config;

    /* We create the main window. */
    main_window = gtk_window_new( GTK_WINDOW_TOPLEVEL );

    /* We set the window size. */
    gtk_widget_set_size_request( GTK_WIDGET( main_window ), width, height );

    /* We set the window title. */
    gtk_window_set_title ( GTK_WINDOW( main_window ), "Genesx" );

    /* We want to leave the application when the window is closed. */
    g_signal_connect ( G_OBJECT( main_window ), "delete_event",
                       G_CALLBACK( gtk_main_quit ), NULL );

    /* We initialize the menus. */
    menu_bar = create_menus( main_window );

    /*
    * We configure an OpenGL-capable visual.
    */

    /* Can we get a double buffered visual? */
    gl_config = gdk_gl_config_new_by_mode ( GDK_GL_MODE_RGB    |
                                            GDK_GL_MODE_DEPTH  |
                                            GDK_GL_MODE_DOUBLE);
    if (gl_config == NULL) {
        g_print ("*** Cannot find the double-buffered visual.\n");
        g_print ("*** Trying single-buffered visual.\n");

        /* Let's try a single buffered one, then. */
        gl_config = gdk_gl_config_new_by_mode ( GDK_GL_MODE_RGB   |
                                                GDK_GL_MODE_DEPTH);
        if (gl_config == NULL) {
            g_print ("*** No appropriate OpenGL-capable visual found.\n");
            exit (1);
        }
    }

    /* Now we create a vertical box which will contain our widgets. */
    vertical_box = gtk_vbox_new (FALSE, 0);

    /* We add the container to our main window... */
    gtk_container_add( GTK_CONTAINER( main_window ), vertical_box);
    /* ...and show it. */
    gtk_widget_show( vertical_box );

    /* Create a menu-bar to hold the menus and add it to our main window */
    gtk_box_pack_start( GTK_BOX( vertical_box ), menu_bar, FALSE, FALSE, 2 );
    gtk_widget_show( menu_bar );

    /* Now we create the drawing area in which we will draw the OpenGL scene. */
    drawing_area = gtk_drawing_area_new ();

    /* We resize it to a suitable size. */
    gtk_widget_set_size_request( drawing_area, width, height );

    /* We set the OpenGL-capability to the widget. */
    gtk_widget_set_gl_capability(   drawing_area,
                                  gl_config,
                                  NULL,
                                  TRUE,
                                  GDK_GL_RGBA_TYPE);

    /* We specify the events that we can get. */
    gtk_widget_add_events( drawing_area,
                           GDK_BUTTON1_MOTION_MASK    |
                           GDK_BUTTON2_MOTION_MASK    |
                           GDK_BUTTON_PRESS_MASK      |
                           GDK_VISIBILITY_NOTIFY_MASK );

    /* Now we connect the signals with the callbacks. */
    g_signal_connect( G_OBJECT( drawing_area ),
                      "map_event",
                      G_CALLBACK( map ),
                      NULL);

    g_signal_connect ( G_OBJECT( drawing_area ),
                       "unmap_event",
                       G_CALLBACK( unmap ),
                       NULL);

    g_signal_connect_swapped( G_OBJECT( main_window ),
                              "key_press_event",
                              G_CALLBACK( on_key_press_event ),
                              drawing_area );

    g_signal_connect( G_OBJECT( drawing_area ),
                      "expose_event",
                      G_CALLBACK( expose_event ),
                      NULL );

    g_signal_connect( G_OBJECT( drawing_area ), "configure_event",
                      G_CALLBACK( reshape ), NULL);

    gtk_box_pack_start( GTK_BOX( vertical_box ), drawing_area, TRUE, TRUE, 0 );
    gtk_widget_show( drawing_area );
    idle_add( drawing_area );

    /* This timer allows us to compute the FPS. */
    if (timer == NULL)
        timer = g_timer_new ();

    g_timer_start (timer);

    /* Show the application window. */
    gtk_widget_show_all( main_window  );

    return;
}

/*
 * This function creates the menus for our application,
 * and returns the menu bar.
 */
static GtkWidget *
create_menus( GtkWidget * main_window ) {
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    gint nmenu_items;
    GtkWidget * menu_bar;

    /* We get the number of menu items. */
    nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);

    /* We create the accelerator group. */
    accel_group = gtk_accel_group_new ();

    /* We create the item factory. */
    item_factory = gtk_item_factory_new( GTK_TYPE_MENU_BAR, "<main>",
                                         accel_group );

    /* We create the items in the item factory. */
    gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

    /* We attach the new accelerator group to the window. */
    gtk_window_add_accel_group (GTK_WINDOW (main_window), accel_group);

    /* We create the menu bar. */
    menu_bar = gtk_item_factory_get_widget (item_factory, "<main>");

    /* And then we return it. */
    return menu_bar;
}

/*
 * This function is called each time an expose_event is received.
 */
static int
expose_event ( GtkWidget      *widget,
               GdkEventExpose *event,
               gpointer        data ) {
    /* We get the context and the drawable. */
    GdkGLContext *gl_context = gtk_widget_get_gl_context( widget );
    GdkGLDrawable *gl_drawable = gtk_widget_get_gl_drawable( widget );
    static int frames = 0;
    double seconds;

    /* Now we begin to draw. */
    if (!gdk_gl_drawable_gl_begin( gl_drawable, gl_context ) )
        return FALSE;
    /* We initialize scene... */
    init_gl( widget->allocation.width, widget->allocation.height );
    /* ...and then draw it. */
    draw_gl_scene();
    /* We update the water position. */
    g_water_pos_counter++;

    /* We swap the buffers. */
    if( gdk_gl_drawable_is_double_buffered( gl_drawable ) )
        gdk_gl_drawable_swap_buffers( gl_drawable );
    else
        glFlush ();

    /* Ok, we're done. */
    gdk_gl_drawable_gl_end( gl_drawable );

    /* One more frame has been displayed. */
    frames++;

    /* Every 5 seconds we display the FPS. */
    if( ( seconds = g_timer_elapsed( timer, NULL ) ) >= 5.0 ) {
        double fps = frames / seconds;
        g_print ("%d frames in %6.3f seconds = %6.3f FPS\n", frames, seconds, fps);
        g_timer_reset (timer);
        frames = 0;
    }

    return TRUE;
}

/*
 * This function is called when the window is resized.
 */
static gboolean
reshape ( GtkWidget         *widget,
          GdkEventConfigure *event,
          gpointer           data ) {
    /* We just resize the GL scene. */
    resize_gl_scene( widget->allocation.width, widget->allocation.height );

    return TRUE;
}

/*
 * This function is called each time a key is pressed.
 */
static gboolean
on_key_press_event( GtkWidget *widget, GdkEventKey *event, gpointer data ) {
    /* Our action depends on the value of the key. */
    switch (event->keyval) {
    case GDK_Right:
        g_theta += g_rotation_step;
        break;
    case GDK_Left:
        g_theta -= g_rotation_step;
        break;
    case GDK_Up:
        g_phi += g_rotation_step;
        break;
    case GDK_Down:
        g_phi -= g_rotation_step;
        break;
    case GDK_Page_Up:
        g_position[0] += g_motion_step * cos( g_theta * PI_OVER_180 );
        g_position[1] += -g_motion_step * sin( g_theta * PI_OVER_180 );
        g_position[2] += g_motion_step * sin( g_phi * PI_OVER_180 );
        break;
    case GDK_Page_Down:
        g_position[0] += -g_motion_step * cos( g_theta * PI_OVER_180 );
        g_position[1] += g_motion_step * sin( g_theta * PI_OVER_180 );
        g_position[2] += -g_motion_step * sin( g_phi * PI_OVER_180 );
        break;
        /* 'w' increases the water level. */
    case GDK_w:
        g_water_level += g_water_level_step;
        break;
        /* 'x' decreases the water level. */
    case GDK_x:
        g_water_level -= g_water_level_step;
        break;
        /* 't' activates or deactivates the textures. */
    case GDK_t:
        g_textures = !g_textures;
        break;
        /* 'f' starts or stops the animation. */
    case GDK_p:
        g_animate = !g_animate;
        break;
        /* 'f' activates or deactivates the fog. */
    case GDK_f:
        g_fog = !g_fog;
        break;
        /* 'g' sets or removes the ground. */
    case GDK_g:
        g_ground = !g_ground;
        break;
        /* 'y' sets or unsets the texture filtering. */
    case GDK_y:
        g_filtered_textures = !g_filtered_textures;
        break;
        /* 'n' activates or deactivates the smooth normals. */
    case GDK_n:
        g_smooth_normals = !g_smooth_normals;
        break;
        /* 'R' activates or deactivates the reflection *EXPERIMENTAL* */
    case GDK_R:
        g_reflection = !g_reflection;
        break;
        /* <escape> quits the application. */
    case GDK_Escape:
        gtk_main_quit();
        break;
    default:
        return TRUE;
    }

    return TRUE;
}

/*
 * This function is called each time an 'idle' event is received.
 */
static gboolean
idle (GtkWidget *widget) {

    if( g_animate ) {
        /* Invalidate the whole window. */
        gdk_window_invalidate_rect (widget->window, &widget->allocation, FALSE);

        /* Update synchronously (fast). */
        if (is_sync)
            gdk_window_process_updates (widget->window, FALSE);

    }

    return TRUE;
}

/*
 * This function adds the idle state to the given widget.
 */
static void
idle_add( GtkWidget *widget ) {
    if ( idle_id == 0 ) {
        idle_id = g_idle_add_full ( GDK_PRIORITY_REDRAW,
                                    ( GSourceFunc ) idle,
                                    widget,
                                    NULL);
    }
}

/*
 * This function removes the idle state from the given widget.
 */
static void
idle_remove( GtkWidget *widget ) {
    if ( idle_id != 0 ) {
        g_source_remove( idle_id );
        idle_id = 0;
    }
}

/*
 * This function is called each time a map event is received.
 */
static gboolean
map( GtkWidget   *widget,
     GdkEventAny *event,
     gpointer     data ) {
    if( g_animate )
        idle_add( widget );
    return TRUE;
}

/*
 * This function is called each time an unmap event is received.
 */
static gboolean
unmap( GtkWidget   *widget,
       GdkEventAny *event,
       gpointer     data ) {
    idle_remove( widget );
    return TRUE;
}

/*
 * This function displays the "New Landscape..." dialog.
 */
int
display_new_landscape_dialog( void ) {
    int i, result = 0;
    GtkWidget *dialog;
    GtkWidget *num_recursions_spin_button, *landscape_size_spin_button,
    *initial_variance_spin_button, *attenuation_factor_spin_button,
    *color_variance_spin_button,
    *landscape_box, *properties_box;
    GtkWidget *horizontal_box;
    Landscape previous_landscape;
    int previous_num_divisions;
    GtkWidget *landscape_color_spin_button[NUM_LEVELS];
    char tmp_string[256];
    GdkColor tmp_color;

    idle_remove( drawing_area );

    dialog = gtk_dialog_new_with_buttons( "New Landscape...",
                                          GTK_WINDOW( main_window ),
                                          GTK_DIALOG_MODAL
                                          | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    gtk_dialog_set_default_response( GTK_DIALOG( dialog ), GTK_RESPONSE_ACCEPT );

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( TRUE, 2 );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ),
                        horizontal_box, TRUE, TRUE, 2 );

    /* Now we create the "Landscape Colors" frame. */
    landscape_box = create_frame( horizontal_box, "Landscape Colors" );

    /* Now we create the "Properties" frame. */
    properties_box = create_frame( horizontal_box, "Properties" );


    for( i = 0 ; i < NUM_LEVELS ; i++ ) {
        sprintf( tmp_string, "Level #%d : ", i );

        landscape_color_spin_button[i] = create_color_button_line( GTK_BOX( landscape_box ),
                                         tmp_string, g_colors_array[i][0], g_colors_array[i][1], g_colors_array[i][2], 0, 0 );
    }

    num_recursions_spin_button = create_spin_button_line( GTK_BOX( properties_box ),
                                 "Number of recursions : ",
                                 g_num_recursions, MIN_RECURSIONS,
                                 MAX_RECURSIONS, 1, 0 );

    landscape_size_spin_button = create_spin_button_line( GTK_BOX( properties_box ),
                                 "Landscape size : ",
                                 g_landscape_size, MINFLOAT,
                                 MAXFLOAT, BIG_STEP, 0 );

    initial_variance_spin_button = create_spin_button_line( GTK_BOX( properties_box ),
                                   "Initial variance : ",
                                   g_initial_variance, MINFLOAT,
                                   MAXFLOAT, BIG_STEP, 0 );

    attenuation_factor_spin_button = create_spin_button_line( GTK_BOX( properties_box ),
                                     "Attenuation factor : ",
                                     g_attenuation_factor, MINFLOAT,
                                     MAXFLOAT, LITTLE_STEP, 3 );

    color_variance_spin_button = create_spin_button_line( GTK_BOX( properties_box ),
                                 "Color variance : ",
                                 g_color_variance, MINFLOAT,
                                 MAXFLOAT, NORMAL_STEP, 2 );

    /* We display all the dialog's widgets. */
    gtk_widget_show_all( dialog );

    /* We run the dialog... */
    if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_ACCEPT ) {
        /* ...if the user did not cancel : */
        /* We free the landscape, if it exists. */
        previous_landscape = g_landscape;
        previous_num_divisions = g_num_divisions;
        g_landscape = NULL;

        if( previous_landscape != NULL )
            free_landscape( previous_landscape, previous_num_divisions );

        /* We store the properties into the global variables. */
        g_num_recursions = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( num_recursions_spin_button ) );
        g_landscape_size = gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( landscape_size_spin_button ) );
        g_initial_variance = gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( initial_variance_spin_button ) );
        g_attenuation_factor = gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( attenuation_factor_spin_button ) );
        g_color_variance = gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( color_variance_spin_button ) );

        for( i = 0 ; i < NUM_LEVELS ; i++ ) {
            gtk_color_button_get_color( GTK_COLOR_BUTTON( landscape_color_spin_button[i] ), &tmp_color );
            g_colors_array[i][0] = tmp_color.red / 65535.0f;
            g_colors_array[i][1] = tmp_color.green / 65535.0f;
            g_colors_array[i][2] = tmp_color.blue / 65535.0f;
        }

        /* Then we generate the landscape. */
        g_landscape = generate_landscape( g_attenuation_factor,
                                          g_num_recursions,
                                          g_initial_variance,
                                          g_landscape_size,
                                          &g_num_divisions,
                                          g_colors_array );
        result = 1;
    }

    gtk_widget_destroy( dialog );

    idle_add( drawing_area );

    return result;
}

/*
 * This function creates a simple frame.
 */
static GtkWidget *
create_frame( GtkWidget * parent, char *caption ) {
    GtkWidget * frame;
    GtkWidget * box;

    /* We create the frame... */
    frame = gtk_frame_new( caption );

    gtk_box_pack_start( GTK_BOX( parent ),
                        frame, TRUE, TRUE, 2 );

    /* ...the box... */
    box = gtk_vbox_new( TRUE, 2 );

    gtk_container_add( GTK_CONTAINER( frame ),
                       box );

    /* ...and return it. */
    return box;
}

/*
* This function displays the "Properties..." dialog.
*/
int
display_properties_dialog( void ) {
    GtkWidget *dialog, *horizontal_box, *first_vertical_box, *second_vertical_box, *options_box,
    *view_box, *water_box, *sky_box, *light_box, *miscellaneous_box,
    *fog_start_spin_button, *fog_end_spin_button,
    *rotation_step_spin_button, *water_level_spin_button, *water_level_step_spin_button,
    *fog_density_spin_button,
    *water_precision_spin_button, *water_opacity_spin_button, *perspective_spin_button, *ground_precision_spin_button,
    *field_of_view_angle_spin_button, *ground_check_button, *smooth_normals_check_button, *filtered_textures_check_button,
    *fog_check_button, *animate_check_button, *textures_check_button,
    *light_x_position_spin_button, *light_y_position_spin_button, *light_z_position_spin_button,
    *sky_color_color_button, *light_ambient_color_color_button, *light_diffuse_color_color_button,
    *motion_step_spin_button, *theta_spin_button, *phi_spin_button, *x_position_spin_button,
    *y_position_spin_button, *z_position_spin_button, *third_vertical_box, *textures_box;
    GtkWidget *landscape_texture_file_chooser_button, *water_texture_file_chooser_button;
    char *file_name;
    int result = 0;
    GdkColor tmp_color;

    idle_remove( drawing_area );

    dialog = gtk_dialog_new_with_buttons( "Properties...",
                                          GTK_WINDOW( main_window ),
                                          GTK_DIALOG_MODAL
                                          | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    gtk_dialog_set_default_response( GTK_DIALOG( dialog ), GTK_RESPONSE_ACCEPT );

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( TRUE, 2 );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ),
                        horizontal_box, TRUE, TRUE, 2 );

    /* Then we create the first vertical box... */
    first_vertical_box = gtk_vbox_new( FALSE, 2 );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        first_vertical_box, TRUE, TRUE, 2 );

    /* Then we create the second vertical box... */
    second_vertical_box = gtk_vbox_new( FALSE, 2 );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        second_vertical_box, TRUE, TRUE, 2 );

    /* Then we create the third vertical box... */
    third_vertical_box = gtk_vbox_new( FALSE, 2 );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        third_vertical_box, TRUE, TRUE, 2 );

    /* Now we create the "Options" frame. */
    options_box = create_frame( first_vertical_box, "Options" );

    /* Now we create the "View" frame. */
    view_box = create_frame( third_vertical_box, "View" );

    /* Now we create the "Water" frame. */
    water_box = create_frame( second_vertical_box, "Water" );

    /* Now we create the "Fog" frame. */
    sky_box = create_frame( first_vertical_box, "Sky" );

    /* Now we create the "Light" frame. */
    light_box = create_frame( second_vertical_box, "Light" );

    /* Now we create the "Miscellaneous" frame. */
    textures_box = create_frame( third_vertical_box, "Textures" );

    /* Now we create the "Miscellaneous" frame. */
    miscellaneous_box = create_frame( third_vertical_box, "Miscellaneous" );

    theta_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                        "Theta : ",
                        g_theta, -MAXFLOAT,
                        MAXFLOAT, NORMAL_STEP, 2 );

    phi_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                      "Phi : ",
                      g_phi, -MAXFLOAT,
                      MAXFLOAT, NORMAL_STEP, 2 );

    x_position_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                             "X position : ",
                             g_position[0], -MAXFLOAT,
                             MAXFLOAT, BIG_STEP, 2 );

    y_position_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                             "Y position : ",
                             g_position[1], -MAXFLOAT,
                             MAXFLOAT, BIG_STEP, 2 );

    z_position_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                             "Z position : ",
                             g_position[2], -MAXFLOAT,
                             MAXFLOAT, BIG_STEP, 2 );

    field_of_view_angle_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                                      "Field of view's angle : ",
                                      g_field_of_view_angle, -MAXFLOAT,
                                      MAXFLOAT, NORMAL_STEP, 2 );

    fog_start_spin_button = create_spin_button_line( GTK_BOX( sky_box ),
                            "Fog start : ",
                            g_fog_start, -MAXFLOAT,
                            MAXFLOAT, BIG_STEP, 2 );

    fog_end_spin_button = create_spin_button_line( GTK_BOX( sky_box ),
                          "Fog end : ",
                          g_fog_end, -MAXFLOAT,
                          MAXFLOAT, BIG_STEP, 2 );

    fog_density_spin_button = create_spin_button_line( GTK_BOX( sky_box ),
                              "Fog density : ",
                              g_fog_density, MINFLOAT,
                              MAXFLOAT, NORMAL_STEP, 2 );

    ground_precision_spin_button = create_spin_button_line( GTK_BOX( miscellaneous_box ),
                                   "Ground quality : ",
                                   g_ground_precision, MINFLOAT,
                                   MAXFLOAT, NORMAL_STEP, 2 );

    light_x_position_spin_button = create_spin_button_line( GTK_BOX( light_box ),
                                   "Light's X position : ",
                                   g_light_position[0], -MAXFLOAT,
                                   MAXFLOAT, BIG_STEP, 2 );

    light_y_position_spin_button = create_spin_button_line( GTK_BOX( light_box ),
                                   "Light's Y position : ",
                                   g_light_position[1], -MAXFLOAT,
                                   MAXFLOAT, BIG_STEP, 2 );

    light_z_position_spin_button = create_spin_button_line( GTK_BOX( light_box ),
                                   "Light's Z position : ",
                                   g_light_position[2], -MAXFLOAT,
                                   MAXFLOAT, BIG_STEP, 2 );

    motion_step_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                              "Motion step : ",
                              g_motion_step, MINFLOAT,
                              MAXFLOAT, NORMAL_STEP, 2 );

    perspective_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                              "Perspective : ",
                              g_perspective, MINFLOAT,
                              MAXFLOAT, BIG_STEP, 2 );

    rotation_step_spin_button = create_spin_button_line( GTK_BOX( view_box ),
                                "Rotation step : ",
                                g_rotation_step, MINFLOAT,
                                MAXFLOAT, NORMAL_STEP, 2 );

    water_level_spin_button = create_spin_button_line( GTK_BOX( water_box ),
                              "Water level : ",
                              g_water_level, -MAXFLOAT,
                              MAXFLOAT, LARGE_STEP, 2 );

    water_level_step_spin_button = create_spin_button_line( GTK_BOX( water_box ),
                                   "Water level step : ",
                                   g_num_recursions, MINFLOAT,
                                   MAXFLOAT, NORMAL_STEP, 2 );

    water_opacity_spin_button = create_spin_button_line( GTK_BOX( water_box ),
                                "Water opacity : ",
                                g_water_opacity, 0,
                                1, LITTLE_STEP, 2 );

    water_precision_spin_button = create_spin_button_line( GTK_BOX( water_box ),
                                  "Water quality : ",
                                  g_water_precision, MINFLOAT,
                                  MAXFLOAT, NORMAL_STEP, 2 );

    sky_color_color_button = create_color_button_line( GTK_BOX( sky_box ),
                             "Sky color :", g_sky_color[0], g_sky_color[1], g_sky_color[2], 0, 0 );

    light_ambient_color_color_button = create_color_button_line( GTK_BOX( light_box ),
                                       "Ambient light color : ",
                                       g_light_ambient_color[0], g_light_ambient_color[1],
                                       g_light_ambient_color[2], /*g_light_ambient_color[3], 1*/0,0 );

    light_diffuse_color_color_button = create_color_button_line( GTK_BOX( light_box ),
                                       "Diffuse light color : ",
                                       g_light_diffuse_color[0], g_light_diffuse_color[1],
                                       g_light_diffuse_color[2], /*g_light_ambient_color[3], 1*/0,0 );

    landscape_texture_file_chooser_button = create_file_chooser_button_line( GTK_BOX( textures_box ),
                                            g_landscape_texture_path, "Landscape Texture :" );

    water_texture_file_chooser_button = create_file_chooser_button_line( GTK_BOX( textures_box ),
                                        g_water_texture_path, "Water Texture :" );

    /* Now we create the options. */
    animate_check_button = create_check_button_line( GTK_BOX( options_box ),
                           "Animate",
                           g_animate );


    filtered_textures_check_button = create_check_button_line( GTK_BOX( options_box ),
                                     "Filtered textures",
                                     g_filtered_textures );

    fog_check_button = create_check_button_line( GTK_BOX( options_box ),
                       "Fog",
                       g_fog );

    ground_check_button = create_check_button_line( GTK_BOX( options_box ),
                          "Ground",
                          g_ground );

    textures_check_button = create_check_button_line( GTK_BOX( options_box ),
                            "Textures",
                            g_textures );

    smooth_normals_check_button = create_check_button_line( GTK_BOX( options_box ),
                                  "Smooth normals",
                                  g_smooth_normals );

    /* We display all the dialog's widgets. */
    gtk_widget_show_all (dialog);

    /* We run the dialog... */
    if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_ACCEPT ) {
        /* ...if the user did not cancel : */
        /* We store the properties into the global variables. */
        g_field_of_view_angle = gtk_spin_button_get_value( GTK_SPIN_BUTTON( field_of_view_angle_spin_button ) );
        g_fog_start = gtk_spin_button_get_value( GTK_SPIN_BUTTON( fog_start_spin_button ) );
        g_fog_end = gtk_spin_button_get_value( GTK_SPIN_BUTTON( fog_end_spin_button ) );
        g_fog_density = gtk_spin_button_get_value( GTK_SPIN_BUTTON( fog_density_spin_button ) );
        g_ground_precision = gtk_spin_button_get_value( GTK_SPIN_BUTTON( ground_precision_spin_button ) );
        g_motion_step = gtk_spin_button_get_value( GTK_SPIN_BUTTON( motion_step_spin_button ) );
        g_perspective = gtk_spin_button_get_value( GTK_SPIN_BUTTON( perspective_spin_button ) );
        g_rotation_step = gtk_spin_button_get_value( GTK_SPIN_BUTTON( rotation_step_spin_button ) );
        g_water_level = gtk_spin_button_get_value( GTK_SPIN_BUTTON( water_level_spin_button ) );
        g_water_level_step = gtk_spin_button_get_value( GTK_SPIN_BUTTON( water_level_step_spin_button ) );
        g_water_opacity = gtk_spin_button_get_value( GTK_SPIN_BUTTON( water_opacity_spin_button ) );
        g_water_precision = gtk_spin_button_get_value( GTK_SPIN_BUTTON( water_precision_spin_button ) );

        g_light_position[0] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( light_x_position_spin_button ) );
        g_light_position[1] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( light_y_position_spin_button ) );
        g_light_position[2] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( light_z_position_spin_button ) );

        g_position[0] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( x_position_spin_button ) );
        g_position[1] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( y_position_spin_button ) );
        g_position[2] = gtk_spin_button_get_value( GTK_SPIN_BUTTON( z_position_spin_button ) );
        g_theta = gtk_spin_button_get_value( GTK_SPIN_BUTTON( theta_spin_button ) );
        g_phi = gtk_spin_button_get_value( GTK_SPIN_BUTTON( phi_spin_button ) );

        gtk_color_button_get_color( GTK_COLOR_BUTTON( sky_color_color_button ), &tmp_color );
        g_sky_color[0] = tmp_color.red / 65535.0f;
        g_sky_color[1] = tmp_color.green / 65535.0f;
        g_sky_color[2] = tmp_color.blue / 65535.0f;

        gtk_color_button_get_color( GTK_COLOR_BUTTON( light_ambient_color_color_button ), &tmp_color );
        g_light_ambient_color[0] = tmp_color.red / 65535.0f;
        g_light_ambient_color[1] = tmp_color.green / 65535.0f;
        g_light_ambient_color[2] = tmp_color.blue / 65535.0f;

        gtk_color_button_get_color( GTK_COLOR_BUTTON( light_diffuse_color_color_button ), &tmp_color );
        g_light_diffuse_color[0] = tmp_color.red / 65535.0f;
        g_light_diffuse_color[1] = tmp_color.green / 65535.0f;
        g_light_diffuse_color[2] = tmp_color.blue / 65535.0f;

        g_animate = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( animate_check_button ) );
        g_ground = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( ground_check_button ) );
        g_filtered_textures = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( filtered_textures_check_button ) );
        g_fog = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( fog_check_button ) );
        g_smooth_normals = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( smooth_normals_check_button ) );

        if( ( file_name = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( landscape_texture_file_chooser_button ) ) ) != NULL ) {
            strcpy( g_landscape_texture_path, file_name );
            g_free( file_name );
        }

        if( ( file_name = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( water_texture_file_chooser_button ) ) ) != NULL ) {
            strcpy( g_water_texture_path, file_name );
            g_free( file_name );
        }

        g_load_textures = 1;
        result = 1;
    }

    gtk_widget_destroy( dialog );

    idle_add( drawing_area );

    return result;
}

/*
 * This function is called when the user chooses "New Landscape..." from
 * the File menu.
 */
static void
on_file_new( GtkWidget *widget, GdkEventKey *event, gpointer data ) {
    if( display_new_landscape_dialog() )
        display_properties_dialog();
}

/*
 * This function is called when the user chooses "Properties..." from the
 * Parameters menu.
 */
static void
on_parameters_properties( GtkBox *widget, GdkEventKey *event, gpointer data ) {
    display_properties_dialog();
}

/*
 * This function creates a line with a file chooser button.
 */
static GtkWidget *
create_file_chooser_button_line( GtkBox *parent, char * initial_path, char *label_text ) {
    GtkWidget *horizontal_box, *label, *alignment, *file_chooser_button;
    GtkFileFilter *filter;

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( FALSE, 2 );

    /* ..and add it to the window. */
    gtk_box_pack_start( GTK_BOX( parent ),
                        horizontal_box, TRUE, TRUE, 2 );

    /* We create the corresponding label... */
    label = gtk_label_new( label_text );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        label, FALSE, FALSE, 2 );

    /* Now we create the color button... */
    file_chooser_button = gtk_file_chooser_button_new ("Select a file", GTK_FILE_CHOOSER_ACTION_OPEN);

    filter = gtk_file_filter_new();
    gtk_file_filter_add_pattern( GTK_FILE_FILTER( filter ), "*.bmp" );
    gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( file_chooser_button ),
                                 GTK_FILE_FILTER( filter ) );

    gtk_file_chooser_select_uri( GTK_FILE_CHOOSER( file_chooser_button ),
                                 initial_path );

    gtk_file_chooser_button_set_width_chars( GTK_FILE_CHOOSER_BUTTON( file_chooser_button ), 10 );

    /* ...its alignment... */
    alignment = gtk_alignment_new( 1, 0.5, 0, 0 );

    gtk_container_add( GTK_CONTAINER ( alignment ), file_chooser_button );

    /* ...and add it. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        alignment, TRUE, TRUE, 2 );

    return file_chooser_button;
}

/*
 * This function creates a line with a color button,
 * and returns the latter.
 */
static GtkWidget *
create_color_button_line( GtkBox *parent, char *label_text,
                          float red, float green, float blue,
                          float alpha, int use_alpha ) {
    GtkWidget *horizontal_box, *label, *color_button, *alignment;
    GdkColor color;

    /* We initialize the color. */
    color.red = red * 65535;
    color.green = green * 65535;
    color.blue = blue * 65535;

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( FALSE, 2 );

    /* ..and add it to the window. */
    gtk_box_pack_start( GTK_BOX( parent ),
                        horizontal_box, TRUE, TRUE, 2 );

    /* We create the corresponding label... */
    label = gtk_label_new( label_text );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        label, FALSE, FALSE, 2 );

    /* Now we create the color button... */
    color_button = gtk_color_button_new_with_color( &color );

    if( use_alpha ) {
        gtk_color_button_set_use_alpha( GTK_COLOR_BUTTON( color_button ), TRUE );
        gtk_color_button_set_alpha( GTK_COLOR_BUTTON( color_button ), alpha * 65536 );
    }
    /* ...its alignment... */
    alignment = gtk_alignment_new( 1, 0.5, 0, 0 );

    gtk_container_add( GTK_CONTAINER ( alignment ), color_button );

    /* ...and add it. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        alignment, TRUE, TRUE, 2 );

    return color_button;
}

/*
 * This function creates a line with a spin button,
 * and returns the latter.
 */
static GtkWidget *
create_spin_button_line( GtkBox *parent, char *label_text,
                         float initial_value, float min_value,
                         float max_value, float step, int digits ) {
    GtkWidget *horizontal_box, *label, *spin_button, *alignment;
    GtkObject *adjustment;

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( FALSE, 2 );

    /* ..and add it to the window. */
    gtk_box_pack_start( GTK_BOX( parent ),
                        horizontal_box, TRUE, TRUE, 5 );

    /* We create the corresponding label... */
    label = gtk_label_new( label_text );

    /* ...and add it to the window. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        label, FALSE, FALSE, 2 );

    /* Now we create the spin button... */
    adjustment = gtk_adjustment_new( ( gdouble )initial_value,
                                     min_value,
                                     max_value,
                                     step, 1, 0 );

    spin_button = gtk_spin_button_new( GTK_ADJUSTMENT( adjustment ), 1, digits );

    /* ...its alignment... */
    alignment = gtk_alignment_new( 1, 0.5, 0, 0 );

    gtk_container_add( GTK_CONTAINER ( alignment ), spin_button );

    /* ...and add it. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        alignment, TRUE, TRUE, 2 );

    return spin_button;
}

/*
 * This function creates a line with a check button in the given container,
 * and returns the latter.
 */
static GtkWidget *
create_check_button_line( GtkBox *parent, char *label_text,
                          float initial_value ) {
    GtkWidget *check_button, *horizontal_box;

    /* We create an horizontal box... */
    horizontal_box = gtk_hbox_new( FALSE, 2 );

    /* ..and add it to the window. */
    gtk_box_pack_start( GTK_BOX( parent ),
                        horizontal_box, TRUE, TRUE, 5 );

    /* Now we create the check button... */
    check_button = gtk_check_button_new_with_mnemonic( label_text );

    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( check_button ),
                                  initial_value );

    /* ...and add it. */
    gtk_box_pack_start( GTK_BOX( horizontal_box ),
                        check_button, TRUE, TRUE, 2 );

    return check_button;
}

/*
 * This function displays the about dialog.
 */
static void
on_help_about( GtkWidget *widget, GdkEventKey *event, gpointer data ) {
    /* The logo. */
    GdkPixbuf * logo;

    /* This array contains the authors. */
    static const char * authors[] = { "Jean-Pierre Lozi <jean-pierre@lozi.org>", NULL
                                    };
    /* This string contains the license. */
    static const char * license =
        "This program is free software; you can redistribute it and/or\n"
        "modify it under the terms of the GNU General Public License\n"
        "as published by the Free Software Foundation; either version 2\n"
        "of the License, or (at your option) any later version.\n"
        "\n"
        "This program is distributed in the hope that it will be useful,\n"
        "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
        "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
        "GNU General Public License for more details.\n"
        "\n"
        "You should have received a copy of the GNU General Public License\n"
        "along with this program; if not, write to the Free Software\n"
        "Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA\n"
        "02110-1301, USA.";

    logo = gdk_pixbuf_new_from_file( "images/logo.png", NULL );

    /* We just show the about dialog. */
    gtk_show_about_dialog(  GTK_WINDOW( main_window ),
                            "name", "Genesx",
                            "version", "0.1",
                            "copyright", "(C) Jean-Pierre Lozi, 2005",
                            "comments", "An OpenGL landscape generator",
                            "license", license,
                            "website", "http://genesx.lozi.org",
                            "website_label", "Genesx Website",
                            "logo", logo,
                            "authors", authors );

}

/*
 * This is called when the user chooses "Open..." from the "File" menu.
 */
static void
on_file_open( GtkWidget *widget, GdkEventKey *event, gpointer data ) {
    GtkWidget *open_dialog;
    GtkFileFilter *filter;
    /* We display the "Open..." dialog. */
    open_dialog = gtk_file_chooser_dialog_new( "Open...", GTK_WINDOW( widget ),
                  GTK_FILE_CHOOSER_ACTION_OPEN,
                  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                  GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                  NULL);

    filter = gtk_file_filter_new();
    gtk_file_filter_add_pattern( GTK_FILE_FILTER( filter ), "*.wrl" );
    gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( open_dialog ),
                                 GTK_FILE_FILTER( filter ) );

    idle_remove( drawing_area );

    /* If the user did not cancel... */
    if( gtk_dialog_run( GTK_DIALOG( open_dialog ) ) == GTK_RESPONSE_ACCEPT ) {
        char *file_name;
        /* ...we get the file chosen by the user... */
        file_name = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( open_dialog ) );
        /* ...and save it. */
        open_file( file_name );

        g_free( file_name );
    }

    /* We destroy the dialog. */
    gtk_widget_destroy( open_dialog );

    idle_add( drawing_area );
}

/*
 * This is called when the user chooses "Save" from the "File" menu.
 */
static void
on_file_save( GtkWidget *widget, GdkEventKey *event, gpointer data ) {

    /* If the file has never been saved, then we call 'Save As...'. */
    if( strlen( g_current_file_path ) == 0 ) {
        on_file_save_as( widget, event, data );
    }

    /* Otherwise, we call the save function from the "files" module. */
    save_file( g_current_file_path );
}

/*
 * This is called when the user chooses "Save as..." from the "File" menu.
 */
static void
on_file_save_as( GtkWidget *widget, GdkEventKey *event, gpointer data ) {
    GtkWidget *save_as_dialog;
    GtkFileFilter *filter;

    /* We display the "Save as..." dialog. */
    save_as_dialog = gtk_file_chooser_dialog_new( "Save as...", GTK_WINDOW( widget ),
                     GTK_FILE_CHOOSER_ACTION_SAVE,
                     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                     GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                     NULL);

    idle_remove( drawing_area );

    filter = gtk_file_filter_new();
    gtk_file_filter_add_pattern( GTK_FILE_FILTER( filter ), "*.wrl" );
    gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( save_as_dialog ),
                                 GTK_FILE_FILTER( filter ) );

    /* If the user did not cancel... */
    if( gtk_dialog_run( GTK_DIALOG( save_as_dialog ) ) == GTK_RESPONSE_ACCEPT ) {
        char *file_name;
        /* ...we get the file chosen by the user... */
        file_name = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( save_as_dialog ) );
        /* ...and save it. */
        save_file( file_name );
        g_free( file_name );
    }

    /* We destroy the dialog. */
    gtk_widget_destroy( save_as_dialog );

    idle_add( drawing_area );
}
