/*****************************************************************************/
/* ========================================================================= */
/*  generation.c                                                             */
/*  The GUI module contains the user interface related functions.            */
/* ========================================================================= */
/*  GenesX - A 3D fractal mountains generator                                */
/*  Copyright (C) Jean-Pierre Lozi, 2005                                     */
/* ========================================================================= */
/*****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*****************************************************************************/
/* Local headers                                                             */
/*****************************************************************************/
#include "tools.h"
#include "constants.h"
#include "globals.h"
#include "generation.h"

/*****************************************************************************/
/* Standard headers                                                          */
/*****************************************************************************/
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
#define min(x,y) (((x)<(y))?(x):(y))
#define max(x,y) (((x)>(y))?(x):(y))

/*****************************************************************************/
/* Function prototypes                                                       */
/*****************************************************************************/

/*
 * Computes the smooth normals of the landscape.
 */
static void
compute_smooth_normals( Landscape landscape, int local_num_divisions );

/*
 * Computes the smooth normal of the triangle, given the normals of
 * the three adjacent sides, a being the side adjacent to the triangle's
 * hypothenuse.
 */
static void
compute_smooth_normal( Vector a, Vector b, Vector c, Vector result );

/*****************************************************************************/
/* Functions                                                                 */
/*****************************************************************************/

/*
 * This function generates the landscape using a fractal algorithm.
 */
Landscape
generate_landscape( float attenuation_factor, int recursions, float variance,
                    float landscape_size, long *num_divisions, Color colors_array[] ) {
    Landscape landscape;
    int i, j, step, double_step, local_num_divisions;
    float cell_size;
    float initial_variance = variance;
    float max_height, min_height;

    /* We initialize the maximum and minimum heights. */
    max_height = -MAXFLOAT;
    min_height = 0;

    /* There are 2^recursions-1 cells on each side. */
    local_num_divisions = ( 1 << recursions ) + 1;

    /* We get the size of the side of each cell (on the (Ox, Oy) axis). */
    cell_size = ( 1.0 * landscape_size ) / local_num_divisions;

    /* We allocate the memory space for our array.*/
    landscape = allocate( sizeof( LandscapePoint * ) * local_num_divisions );

    for( i = 0 ; i < local_num_divisions ; i++ ) {
        landscape[i] = allocate( sizeof( LandscapePoint ) * local_num_divisions );
    }

    /* The points on the border of the grid have a height of 0. */
    for( i = 0 ; i < local_num_divisions ; i++ ) {
        landscape[i][0].position[2] = 0;
        landscape[i][local_num_divisions - 1].position[2] = 0;
        landscape[0][i].position[2] = 0;
        landscape[local_num_divisions - 1][i].position[2] = 0;
    }

    /*
     * We use an iterative algorithm instead of a recursive one, to improve
     * the generation speed. We "jump" from cells to cells with a step that
     * we divide by two at each iteration. 
     * 
     * The algorithm works in the following way : We have a square on the
     * (Ox, Oy) axis and we try to divide it into 4 squares. We evaluate
     * the height of the new points and add a random value to them.
     */
    for( step = local_num_divisions / 2 ; step > 0 ; step = step / 2 ) {
        /* We store this in order to avoid to compute the same operation
         * thousands of times. */
        double_step = step * 2;

        /* Now we're computing the points marked with an x in the following
         * example :
         *
         * o + o + o
         * x - x - x
         * o + o + o
         * x - x - x
         * o + o + o
         * 
         * We force the most-left and most-right points to have a height
         * of 0, for obvious reasons.
         */
        for( i = step ; i <= local_num_divisions - step - 1; i += double_step ) {
            for( j = double_step ;
                    j <= local_num_divisions - double_step - 1;
                    j += double_step ) {
                /* We use a linear interpolation... */
                landscape[i][j].position[2] =
                    ( landscape[i-step][j].position[2]
                      + landscape[i+step][j].position[2] ) / 2;

                /* ...then we randomize the height. */
                landscape[i][j].position[2] += gauss_random( variance );

                if( landscape[i][j].position[2] > max_height )
                    max_height = landscape[i][j].position[2];
            }
        }

        /* Now we're computing the points marked with a + in the following
         * example :
         *
         * o + o + o
         * x - x - x
         * o + o + o
         * x - x - x
         * o + o + o
         * 
         * We force the most-left and most-right points to have an height
         * of 0, for obvious reasons.
         */
        for( i = double_step ;
                i <= local_num_divisions - double_step - 1;
                i += double_step ) {
            for( j = step ; j <= local_num_divisions - step - 1 ; j += double_step ) {
                /* We use a linear interpolation... */
                landscape[i][j].position[2] =
                    ( landscape[i][j-step].position[2]
                      + landscape[i][j+step].position[2] ) / 2;

                /* ...then we add a random value to the height. */
                landscape[i][j].position[2] += gauss_random( variance );

                if( landscape[i][j].position[2] > max_height )
                    max_height = landscape[i][j].position[2];
            }
        }

        /* Now we're computing the points marked with a - in the following
        * example :
        *
        * o + o + o
        * x - x - x
        * o + o + o
        * x - x - x
        * o + o + o
        */
        for( i = step ; i <= local_num_divisions - step - 1 ; i += double_step ) {
            for( j = step ; j <= local_num_divisions - step - 1 ; j += double_step ) {
                /* We use a linear interpolation. */
                landscape[i][j].position[2] =
                    ( landscape[i-step][j-step].position[2]
                      + landscape[i+step][j+step].position[2]
                      + landscape[i-step][j+step].position[2]
                      + landscape[i+step][j-step].position[2] )/ 4;

                /* We don't want an underwater scene! Therefore we want the
                 * first value of the random parameter to be positive. */
                if( variance - initial_variance < 1e-6 )
                    landscape[i][j].position[2] += fabs( gauss_random( variance ) );
                else
                    landscape[i][j].position[2] += gauss_random( variance );

                if( landscape[i][j].position[2] > max_height )
                    max_height = landscape[i][j].position[2];
            }
        }

        /* Then we update the step value. */
        variance = variance / g_attenuation_factor;
    }
    /* The points on the border of the grid have a height of 0. */
    for( i = 0 ; i < local_num_divisions ; i++ ) {
        landscape[i][0].position[2] = 0;
        landscape[i][local_num_divisions - 1].position[2] = 0;
        landscape[0][i].position[2] = 0;
        landscape[local_num_divisions - 1][i].position[2] = 0;
    }

    compute_landscape_properties( 1, landscape, local_num_divisions,
                                  cell_size, landscape_size, min_height,
                                  max_height, colors_array );

    /* We set the num_divisions variable... */
    *num_divisions = local_num_divisions;

    /* ...and return the landscape. */
    return landscape;
}

/*
 * This function computes the landscape's properties.
 */
void
compute_landscape_properties( int compute_colors, Landscape landscape, int num_divisions,
                              float cell_size, int landscape_size,
                              float min_height, float max_height,
                              Color colors_array[] ) {
    Vector top_left, top_right, bottom_left, bottom_right;
    int i, j;
    Vector first_vector, second_vector;
    float level;
    float level_decimal_part;
    
    /*
     * Now we fill in the other fields of the landscape points, ie
     * the x and y coordinates, the normals, and the level.
     */

    /* We center the landscape. */
    bottom_left[0] = top_left[0] = 0 - landscape_size / 2;
    bottom_right[0] = top_right[0] = cell_size - landscape_size / 2;
    top_right[1] = top_left[1] = 0 + landscape_size / 2;
    bottom_right[1] = bottom_left[1] = -cell_size + landscape_size / 2;

    /* For each point... */
    for( i = 0 ; i < num_divisions - 1 ; i++ ) {
        for( j = 0 ; j < num_divisions - 1 ; j++ ) {

            /* We use these vectors to have a more readable code. */
            top_left[2] = landscape[i][j].position[2];
            top_right[2] = landscape[i][j + 1].position[2];
            bottom_left[2] = landscape[i + 1][j].position[2];
            bottom_right[2] = landscape[i + 1][j + 1].position[2];

            /* We update the point's x and y coordinates. */
            landscape[i][j].position[0] = top_left[0];
            landscape[i][j].position[1] = top_left[1];

            if( compute_colors ) {
                /* We compute the "level" of the point : it depends on its height.
                 * We use the following scale : the colors
                 * change from min_height to max_height. */
                level = landscape[i][j].position[2] + gauss_random( g_color_variance );
                level = max( level, min_height );
                level = ( ( level - min_height ) / ( max_height - min_height ) ) * ( NUM_LEVELS - 2 );
                level_decimal_part = level - floor( level );

                /* No we compute the local color of the point. */
                landscape[i][j].color[0] =
                    ( 1 - level_decimal_part ) * colors_array[( int )level][0]
                    + level_decimal_part * colors_array[( int )level + 1][0];
                landscape[i][j].color[1] =
                    ( 1 - level_decimal_part ) * colors_array[( int )level][1]
                    + level_decimal_part * colors_array[( int )level + 1][1];
                landscape[i][j].color[2] =
                    ( 1 - level_decimal_part ) * colors_array[( int )level][2]
                    + level_decimal_part * colors_array[( int )level + 1][2];
            }

            /* Each point has two normals, for it has two adjacent triangles. */
            difference( top_right, top_left, first_vector ),
            difference( bottom_left, top_left, second_vector );
            cross_product( first_vector, second_vector, landscape[i][j].bottom_right_normal );

            /* The normal is in the opposite direction. */
            landscape[i][j].bottom_right_normal[0] =
                -landscape[i][j].bottom_right_normal[0];
            landscape[i][j].bottom_right_normal[1] =
                -landscape[i][j].bottom_right_normal[1];
            landscape[i][j].bottom_right_normal[2] =
                -landscape[i][j].bottom_right_normal[2];

            /* Second normal. */
            difference( top_right, bottom_right, first_vector ),
            difference( bottom_left, bottom_right, second_vector );
            cross_product( first_vector, second_vector, landscape[i + 1][j + 1].top_left_normal );

            /* No we compute the next coordinates. */
            top_right[0] += cell_size;
            top_left[0] += cell_size;
            bottom_right[0] += cell_size;
            bottom_left[0] += cell_size;
        }
        bottom_left[0] = top_left[0] = 0 - landscape_size / 2;
        bottom_right[0] = top_right[0] = cell_size - landscape_size / 2;

        top_right[1] -= cell_size;
        top_left[1] -= cell_size;
        bottom_right[1] -= cell_size;
        bottom_left[1] -= cell_size;
    }

    /* The last two lines have to be computed */
    for( i = 0 ; i < num_divisions ; i++ ) {
        landscape[i][num_divisions - 1].position[0] = landscape[i][num_divisions - 2].position[0] + cell_size;
        landscape[i][num_divisions - 1].position[1] = landscape[i][num_divisions - 2].position[1] + cell_size;
        landscape[num_divisions - 1][i].position[0] = landscape[num_divisions - 2][i].position[0] + cell_size;
        landscape[num_divisions - 1][i].position[1] = landscape[num_divisions - 2][i].position[1] - cell_size;
        landscape[i][num_divisions - 1].color[0] = landscape[num_divisions - 1][i].color[0] = g_colors_array[0][0];
        landscape[i][num_divisions - 1].color[1] = landscape[num_divisions - 1][i].color[1] = g_colors_array[0][1];
        landscape[i][num_divisions - 1].color[2] = landscape[num_divisions - 1][i].color[2] = g_colors_array[0][2];
    }

    /* Now we compute the smooth normals. */
    compute_smooth_normals( landscape, num_divisions );
}

/*
 * This function frees the current landscape.
 */
void
free_landscape( Landscape landscape, int num_divisions ) {
    int i;

    for( i = 0 ; i < num_divisions ; i++ )
        free( landscape[i] );

    free( landscape );
}

/*
 * Computes the smooth normals of the landscape.
 */
static void
compute_smooth_normals( Landscape landscape, int local_num_divisions ) {
    int i, j;
    Vector vertical_normal = { 0.0f, 0.0f, 1.0f };

    /*
     * If we want a smooth rendering, we need, for each normal to set the
     * average of the neighbouring facets' normal as facet's smooth normal.
     */
    for( i = 1 ; i < local_num_divisions - 1 ; i++ ) {
        for( j = 1 ; j < local_num_divisions - 1; j++ ) {
            compute_smooth_normal(  landscape[i][j].top_left_normal,
                                    landscape[i-1][j].bottom_right_normal,
                                    landscape[i][j-1].bottom_right_normal,
                                    landscape[i][j].top_left_smooth_normal );

            compute_smooth_normal( landscape[i][j].bottom_right_normal,
                                   landscape[i+1][j].top_left_normal,
                                   landscape[i][j+1].top_left_normal,
                                   landscape[i][j].bottom_right_smooth_normal );
        }
    }

    /* Now we compute the normals for the last two lines. */
    for( i = 0 ; i < local_num_divisions ; i++ ) {
        memcpy( landscape[i][local_num_divisions - 1].top_left_smooth_normal,
                vertical_normal, sizeof( Vector ) );
        memcpy( landscape[i][local_num_divisions - 2].bottom_right_smooth_normal,
                vertical_normal, sizeof( Vector ) );
        memcpy( landscape[local_num_divisions - 1][i].top_left_smooth_normal,
                vertical_normal, sizeof( Vector ) );
        memcpy( landscape[local_num_divisions - 1][i].bottom_right_smooth_normal,
                vertical_normal, sizeof( Vector ) );
        memcpy( landscape[i][0].bottom_right_smooth_normal,
                vertical_normal, sizeof( Vector ) );
        memcpy( landscape[i][1].top_left_smooth_normal,
                vertical_normal, sizeof( Vector ) );
    }
}

/*
 * Computes the smooth normal of the triangle, given the normals of
 * the three adjacent sides.
 */
static void
compute_smooth_normal( Vector a, Vector b, Vector c, Vector result ) {
    float length;

    result[0] = ( c[0] + a[0] );
    result[1] = ( c[1] + a[1] );
    result[2] = ( c[2] + a[2] );

    length = sqrt( result[0] * result[0] + result[1] * result[1] +
                   result[2] * result[2] );

    result[0] /= length;
    result[1] /= length;
    result[2] /= length;

}
