/*****************************************************************************/
/* ========================================================================= */
/*  files.c                                                                  */
/*  The files module contains the file related functions.                    */
/* ========================================================================= */
/*  GenesX - A 3D fractal mountains generator                                */
/*  Copyright (C) Jean-Pierre Lozi, 2005                                     */
/* ========================================================================= */
/*****************************************************************************/

/*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU Library General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*****************************************************************************/
/* Standard headers                                                          */
/*****************************************************************************/
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/*****************************************************************************/
/* Local headers                                                             */
/*****************************************************************************/
#include "constants.h"
#include "generation.h"
#include "globals.h"

/*****************************************************************************/
/* Gtk+ headers                                                              */
/*****************************************************************************/
#include <gtk/gtk.h>

/*****************************************************************************/
/* Constants                                                                 */
/*****************************************************************************/
#define MAX_LINE_SIZE 128

/*****************************************************************************/
/* Private functions                                                         */
/*****************************************************************************/
/*
* This function opens a file.
*/
static FILE *
open( char * file_name, char * mode );


/*
 * This function is called when an error occurs while reading the file.
 */
static void
file_open_error( void );

/*****************************************************************************/
/* Functions                                                                 */
/*****************************************************************************/

/*
* Saves the current landscape to a VRML file.
*/
void
save_file( char * file_name ) {
    int i, j;
    FILE *file;
    char *old_locale;

    /* We want dots when our floating-point numbers are printed. */
    old_locale = setlocale( LC_NUMERIC, "C" );

    file = open( file_name, "w" );

    /* We write the header. */
    fprintf( file, "#VRML V2.0 utf8\n" );

    /* We write the comments. */
    fprintf( file, "\n#Generated by Genesx 0.1, (C) Jean-Pierre Lozi, 2005\n\n" );

    /* We create a viewpoint which allows us to see the whole scene. */
    fprintf( file, "Viewpoint {\n" );
    fprintf( file, "\tposition %f %f %f\n", g_landscape_size / 2, 0.0f, g_landscape_size * 1.5 );
    fprintf( file, "\tdescription \"View\"\n}\n" );

    fprintf( file, "DEF Landscape Shape {\n" );
    fprintf( file, "\tgeometry ElevationGrid {\n" );
    fprintf( file, "\tcreaseAngle 3.14\n" );
    fprintf( file, "\tsolid FALSE\n" );
    fprintf( file, "\tcolorPerVertex TRUE\n" );
    fprintf( file, "\txDimension %ld\n", g_num_divisions );
    fprintf( file, "\tzDimension %ld\n", g_num_divisions );
    fprintf( file, "\txSpacing %f\n", g_landscape_size / g_num_divisions );
    fprintf( file, "\tzSpacing %f\n", g_landscape_size / g_num_divisions );

    /* Now we start writing the heights. */
    fprintf( file, "\t\theight[\n" );

    fprintf( file, "\t\t" );

    for( i = 0 ; i < g_num_divisions ; i++ ) {
        for( j = 0 ; j < g_num_divisions ; j++ ) {
            fprintf( file, "%f ", g_landscape[i][j].position[2] );
        }
    }

    fprintf( file, "\n\t\t]\n" );
    fprintf( file, "\t\tcolor Color {\n" );
    fprintf( file, "\t\t\tcolor[\n" );

    fprintf( file, "\t\t\t\t" );

    /* Now we start writing the colors. */
    for( i = 0 ; i < g_num_divisions ; i++ ) {
        for( j = 0 ; j < g_num_divisions ; j++ ) {
            fprintf( file, "%f %f %f ", g_landscape[i][j].color[0],
                     g_landscape[i][j].color[1],
                     g_landscape[i][j].color[2] );
        }
    }

    fprintf( file, "\n\t\t\t]\n" );
    fprintf( file, "\t\t}\n" );
    fprintf( file, "\t}\n" );
    fprintf( file, "}\n" );

    setlocale( LC_NUMERIC, old_locale );

    fclose( file );

    /* We're done :) */
    return;
}

/*
* Opens the given VRML file.
*/
void
open_file( char * file_name ) {
    int i, j;
    Landscape previous_landscape;
    int previous_num_divisions;
    char line[MAX_LINE_SIZE];
    float spacing;
    FILE *file;
    char *old_locale;
    float max_height;

    max_height = -MAXFLOAT;

    /* We free the landscape, if it exists. */
    previous_landscape = g_landscape;
    previous_num_divisions = g_num_divisions;
    g_landscape = NULL;

    if( previous_landscape != NULL )
        free_landscape( previous_landscape, previous_num_divisions );


    /* We want dots when our floating-point numbers are scanned. */
    old_locale = setlocale( LC_NUMERIC, "C" );

    file = open( file_name, "r" );

    /* We skip the 13 first lines. */
    for( i = 0 ; i < 13 ; i++ )
        if( fgets( line, MAX_LINE_SIZE, file ) == NULL ) {
            file_open_error();
            return;
        }

    /* We skip the 12 first letters. */
    for( i = 0 ; i < 12 ; i++ )
        if( fgetc( file ) == EOF ) {
            file_open_error();
            return;
        }

    /* We read the number of divisions. */
    if( fscanf( file, "%ld", &g_num_divisions ) != 1 ) {
        file_open_error();
        return;
    }

    /* We skip the next 2 lines. */
    for( i = 0 ; i < 2 ; i++ )
        if( fgets( line, MAX_LINE_SIZE, file ) == NULL ) {
            file_open_error();
            return;
        }

    /* We skip the 10 first letters. */
    for( i = 0 ; i < 10 ; i++ )
        if( fgetc( file ) == EOF ) {
            file_open_error();
            return;
        }

    /* We read the spacing. */
    if( fscanf( file, "%f", &spacing ) != 1 ) {
        file_open_error();
        return;
    }

    g_landscape_size = spacing * g_num_divisions;

    /* We skip the 3 next lines. */
    for( i = 0 ; i < 3 ; i++ )
        if( fgets( line, MAX_LINE_SIZE, file ) == NULL ) {
            file_open_error();
            return;
        }

    /* We allocate the memory space for our array.*/
    g_landscape = allocate( sizeof( LandscapePoint * ) * g_num_divisions );

    for( i = 0 ; i < g_num_divisions ; i++ ) {
        g_landscape[i] = allocate( sizeof( LandscapePoint ) * g_num_divisions );
    }

    /* We fill the heights. */
    for( i = 0 ; i < g_num_divisions ; i++ ) {
        for( j = 0 ; j < g_num_divisions ; j++ ) {
            if( fscanf( file, "%f", &g_landscape[i][j].position[2] ) != 1 ) {
                file_open_error();
                return;
            }
            if( g_landscape[i][j].position[2] > max_height )
                max_height = g_landscape[i][j].position[2];
        }
    }

    /* We skip the next 4 lines. */
    for( i = 0 ; i < 4 ; i++ )
        if( fgets( line, MAX_LINE_SIZE, file ) == NULL ) {
            file_open_error();
            return;
        }

    /* We fill the colors. */
    for( i = 0 ; i < g_num_divisions ; i++ ) {
        for( j = 0 ; j < g_num_divisions ; j++ ) {
            if( fscanf( file, "%f %f %f", &g_landscape[i][j].color[0],
                        &g_landscape[i][j].color[1],
                        &g_landscape[i][j].color[2] ) != 3 ) {
                printf( "%f     %f    %f\n", g_landscape[i][j].color[0],
                        g_landscape[i][j].color[1],
                        g_landscape[i][j].color[2]  );
                file_open_error();
                return;
            }
        }
    }

    /* Then we compute the landscape properties. */
    compute_landscape_properties( 0, g_landscape, g_num_divisions,
                                  spacing, g_landscape_size,
                                  0, max_height,
                                  g_colors_array );

    setlocale( LC_NUMERIC, old_locale );

    fclose( file );

    /* We're done :) */
    return;
}

/*
 * This function is called when an error occurs while reading the file.
 */
static void
file_open_error( void ) {
    GtkWidget *dialog;
    g_landscape = NULL;

    /* We display an error message. */
    dialog = gtk_message_dialog_new ( NULL,
                                      GTK_DIALOG_DESTROY_WITH_PARENT,
                                      GTK_MESSAGE_ERROR,
                                      GTK_BUTTONS_CLOSE,
                                      "Error opening the file." );

    gtk_dialog_run( GTK_DIALOG( dialog ) );
    gtk_widget_destroy( GTK_WIDGET( dialog ) );
}

/*
 * This function opens a file.
 */
static FILE *
open( char * file_name, char * mode ) {
    FILE *result;

    if( ( result = fopen( file_name, mode ) ) == NULL ) {
        fprintf( stderr, "Could not open file : %s", file_name );
        exit( EXIT_FAILURE );
    }

    return result;
}
