package org.lozi.ajp.ajp;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;

import javax.swing.JFrame;
import net.java.games.jogl.*; 

/**
 * AjpUserWindow class.
 * AjpUserWindow's objects represent the Ajp application's windows.
 * 
 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
 */
public class AjpUserWindow {
	/** The corresponding JFrame. */
	private JFrame frame;
	/** The event listener. */
	private AjpGLEventListener listener;
	/** The canvas. */
	private GLCanvas context;
	/** The animator. */
	private Animator animator;
	
	/**
	 * Creates an AjpUserWindow.
	 * 
	 * @param JFrame The corresponding JFrame.
	 * @param listener The window listener.
	 * @param firstLight The first light.
	 * @param secondLight The second light.
	 */
	AjpUserWindow( JFrame frame, AjpGLEventListener listener, GLCanvas context ) {
		this.frame = frame;
		this.listener = listener;
		this.context = context;
		this.listener.setWindow( this );
		animator = new Animator(context);
		
		AjpApp.addFrame( this );
		
		frame.addWindowListener(new AjpUserWindowAdapter( animator ) );
	}
	
	
	
	/**
	 * Animates the window.
	 */
	public void animate () {
		 animator.start();
	}
	
	
	/**
	 * Gets the list of the parametered curves bound to this window.
	 */
	public ArrayList<AjpUserParameteredCurve> getParameteredCurvesArrayList() {
		return listener.getParameteredCurvesArrayList();
	}
	
	/**
	 * Gets the list of the parametered surfaces bound to this window.
	 */
	public ArrayList<AjpUserParameteredSurface> getParameteredSurfacesArrayList() {
		return listener.getParameteredSurfacesArrayList();
	}
	
	/**
	 * Gets the list of the surfaces bound to this window.
	 */
	public ArrayList<AjpUserSurface> getSurfacesArrayList() {
		return listener.getSurfacesArrayList();
	}
	
	/**
	 * Gets the list of the dots bound to this window.
	 */
	public ArrayList<AjpUserDot> getDotsArrayList() {
		return listener.getDotsArrayList();
	}
	
	/**
	 * Rotates the window contents.
	 * 
	 * @param leftRightStep The left/right step.
	 * @param topBottomStep The top/bottom step.
	 */
	public void setRotationSteps( double leftRightStep, double topBottomStep ) {
		this.listener.setLeftRightRotationStep( leftRightStep );
		this.listener.setTopBottomRotationStep( topBottomStep );
	}
	
	/**
	 * Sets the bounds of the window's axes.
	 * 
	 * @param xLowerBound The x lower bound.
	 * @param xUpperBound The x upper bound.
	 * @param yLowerBound The y lower bound.
	 * @param yUpperBound The y upper bound.
	 * @param zLowerBound The z lower bound.
	 * @param zUpperBound The z upper bound.
	 */
	public void setAxesBounds(	double xLowerBound, double xUpperBound, double yLowerBound, double yUpperBound,
								double zLowerBound, double zUpperBound ) {
		listener.setBounds( xLowerBound, xUpperBound, yLowerBound, yUpperBound, zLowerBound, zUpperBound );
	}
	
	/**
	 * Sets the axes' color.
	 * 
	 * @param color The new axes' color.
	 */
	public void setAxesColor( AjpUserColor color ) {
		listener.setAxesColor( color );
	}
	
	/**
	 * Shows/Hides the window.
	 *
	 * @param visibility True to show the window, false to hide it.
	 */
	public void setVisible( boolean visibility ) {
		this.frame.setVisible( visibility );
	}
	
	/**
	 * Sets the background color.
	 * 
	 * @param color The new background color.
	 */
	public void setBackgroundColor( AjpUserColor color ) {
		listener.setBackgroundColor( color );
	}	
	
	/**
	 * Sets the list of the parametered curves bound to this window.
	 * 
	 * @param parameteredCurvesArrayList The list of parametered curves to set.
	 */
	public void setParameteredCurvesArrayList( ArrayList<AjpUserParameteredCurve> parameteredCurvesArrayList ) {
		listener.setParameteredCurvesArrayList( parameteredCurvesArrayList );
	}
	
	/**
	 * Sets the list of the parametered surfaces bound to this window.
	 * 
	 * @param parameteredSurfacesArrayList The list parametered surfaces to set.
	 */
	public void setParameteredSurfacesArrayList( ArrayList<AjpUserParameteredSurface> parameteredSurfacesArrayList ) {
		listener.setParameteredSurfacesArrayList( parameteredSurfacesArrayList );
	}
	
	/**
	 * Sets the list of the surfaces bound to this window.
	 * 
	 * @param surfacesArrayList The list of surfaces to set.
	 */
	public void setSurfacesArrayList( ArrayList<AjpUserSurface> surfacesArrayList ) {
		listener.setSurfacesArrayList( surfacesArrayList );
	}
	
	/**
	 * Binds a parametered curve to the current window.
	 * 
	 * @param curve The parametered curve to bind.
	 */
	public void addParameteredCurve( AjpUserParameteredCurve curve  ) {
		listener.addParameteredCurve( curve );
	}
	
	/**
	 * Binds a parametered surface to the current window.
	 * 
	 * @param curve The parametered curve to bind.
	 */
	public void addParameteredSurface( AjpUserParameteredSurface surface  ) {
		listener.addParameteredSurface( surface );
	}
	
	/**
	 * Binds a surface to the current window.
	 * 
	 * @param surface The surface to bind.
	 */
	public void addSurface( AjpUserSurface surface ) {
		listener.addSurface( surface );
	}
	
	/**
	 * Binds a dot to the current window.
	 * 
	 * @param surface The surface to bind.
	 */
	public void addDot( AjpUserDot dot ) {
		listener.addDot( dot );
	}
	
	/**
	 * Sets the initial camera position and direction.
	 * 
	 * @param Ajp3DUserPoint position The camera position.
	 * @param xRot The x initial rotation.
	 * @param yDir The y initial rotation.
	 * @param zDir The z initial rotation.
	 */
	public void setInitialCameraPositionAndDirection( AjpUser3DPoint position, double xRot, double yRot, double zRot ) {
		listener.setInitialCameraPositionAndDirection( position, xRot, yRot, zRot );
	}
	
	
	/**
	 * Set the window's action.
	 * 
	 * @param action The action to bind.
	 */
	public void setAction( AjpUserWindowAction action ) {
		listener.setAction( action );
	}
	
	/**
	 * @return Returns the listener.
	 */
	AjpGLEventListener getListener() {
		return listener;
	}

	/**
	 * @param listener The listener to set.
	 */
	void setListener(AjpGLEventListener listener) {
		this.listener = listener;
	}
	
	/**
	 * Closes the window.
	 */
	void close () {
		animator.stop();
		frame.setVisible( false );
	}
	
	/**
	 * The AjpUserWindowAdapter which stops the animation when the window is closed.
	 * 
	 * @author jeep - jean-pierre@lozi.org
	 *
	 */
	public class AjpUserWindowAdapter extends WindowAdapter {
		/** The animator. */
		private Animator animator;
		
		/**
		 * The constructor.
		 * 
		 * @param animator The window's animator.
		 */
		public AjpUserWindowAdapter	( Animator animator ) {
			this.animator = animator;
		}
		
		/**
		 * This function is called when the window is close.
		 * 
		 * @param winEvt The window event.
		 */
	    public void windowClosing(WindowEvent winEvt) {
	    		animator.stop();
	    }
	}
}