package org.lozi.ajp.ajp;

import javax.swing.*;
import java.awt.event.*;
import java.util.logging.Logger;
import java.net.URL;

/**
 * AjpUI class. The graphical user interface.
 * 
 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
 * @version 1.0
 */
public class AjpGUI extends JFrame
{
	/**
	 * The serial Version UID.
	 */
	private static final long serialVersionUID = 1L;
	/** We store the fileChooser in order to always start from the previously open directory. */
	private JFileChooser fileChooser;
	/** We keep a pointer to the application. */
	private AjpApp application;
	
	
	public void init( AjpApp application ) {
		Logger logger = Logger.getLogger("AjpGUI");
		logger.info("Entering the AjpGUI constructor");

		this.application = application;
		
		setTitle( "Ajp - Another Java Plotter" );
		// We initialize the file chooser, with the base directory. */
		fileChooser = new JFileChooser( AjpFileTools.baseFolder() );
		
		JFrame.setDefaultLookAndFeelDecorated(false);
		
		// We create the actions.
		LoadSourceAction loadSourceAction = new LoadSourceAction();
		LoadClassAction loadClassAction = new LoadClassAction();
		StartAction startAction = new StartAction();
		StopAction stopAction = new StopAction();
		QuitAction quitAction = new QuitAction();
		
		// We create the menu bar. 
		JMenuBar menuBar = new JMenuBar();
		// We create the File menu.
		JMenu fileMenu = new JMenu( "File" );
		menuBar.add( fileMenu );
		// We create the menu items for the File menu.
		fileMenu.add( new JMenuItem( loadSourceAction ) );
		fileMenu.add( new JMenuItem( loadClassAction ) );
		fileMenu.addSeparator();
		fileMenu.add( new JMenuItem( startAction ) );
		fileMenu.add( new JMenuItem( stopAction ) );
		fileMenu.addSeparator();
		fileMenu.add( new JMenuItem( quitAction ) );
		// We create the toolBar.
		JToolBar toolBar = new JToolBar();
		toolBar.add( loadSourceAction );
		toolBar.add( loadClassAction );
		toolBar.addSeparator();
		toolBar.add( startAction );
		toolBar.add( stopAction );
		toolBar.addSeparator();
		toolBar.add( quitAction );
		
		this.setJMenuBar( menuBar );
		this.getContentPane().add( toolBar );
		
		this.pack();
		this.getContentPane().repaint();
		this.setVisible( true );
		logger.info("Leaving the AjpGUI constructor");
	}
	
	/**
	 * The "Load source" action class.
	 */
	class LoadSourceAction extends AbstractAction {

		/**
		 * The serial version UID.
		 */
		private static final long serialVersionUID = 153572L;
		
		
		/**
		 * Creates a LoadSource object, with its default values.
		 */
		public LoadSourceAction () {
			// We call the constructor wit its default values.
			this( "Load Source...",  new ImageIcon( AjpFileTools.getImage( "source.png", AjpFileTools.IMAGE_TYPE_ICON ) ), "Loads a java source file.", new Integer( 'S' ));
		}
		
		/**
		 * Creates a LoadSourceAction object.
		 * 
		 * @param text The text.
		 * @param icon The icon.
		 * @param desc The description.
		 * @param mnemonic The mnemonic.
		 */
	    public LoadSourceAction ( String text, ImageIcon icon,
	                       String desc, Integer mnemonic) {
	    	// We call the parent constructor.
	    	super(text, icon);
	    	// We set the description and the mnemonic.
	        putValue(SHORT_DESCRIPTION, desc);
	        putValue(MNEMONIC_KEY, mnemonic);
	    }
	    
	    /**
	     * This function is called when the Load Source action is performed.
	     */
	    public void actionPerformed(ActionEvent e) {
	    	
	    	// We display the file chooser.
	    	int returnValue = fileChooser.showOpenDialog( /*AjpGUI.this*/ null );
	    	
	    	// If the user cancelled, then we just return.
	    	if( returnValue != JFileChooser.APPROVE_OPTION ) return;
	    	
	    	// Otherwiste, we load the file in the JEditorPane...
	    	// TODO : load the file in the JEditorPane.
	    	
	    	// ...and we ask the application to load it.
	    	application.loadSource( fileChooser.getSelectedFile() );
	    }
	}
	
	/**
	 * The "Load class" action class.
	 */
	class LoadClassAction extends AbstractAction {

		/**
		 * The serial version UID.
		 */
		private static final long serialVersionUID = 153572L;
		
		
		/**
		 * Creates a LoadClass object, with its default values.
		 */
		public LoadClassAction () {
			this("Load Class...", new ImageIcon( AjpFileTools.getImage( "class.png", AjpFileTools.IMAGE_TYPE_ICON ) ), "Loads a java class.", new Integer( 'O' ));
		}
		
		/**
		 * Creates a LoadClass object.
		 * 
		 * @param text The text.
		 * @param icon The icon.
		 * @param desc The description.
		 * @param mnemonic The mnemonic.
		 */
	    public LoadClassAction ( String text, ImageIcon icon,
	                       String desc, Integer mnemonic) {
	    	// We call the parent constructor.
	    	super(text, icon);
	    	// We set the description and the mnemonic.
	        putValue(SHORT_DESCRIPTION, desc);
	        putValue(MNEMONIC_KEY, mnemonic);
	    }
	    
	    /**
	     * This function is called when the Load Source action is performed.
	     */
	    public void actionPerformed(ActionEvent e) {
	    	// We display the file chooser.
	    	int returnValue = fileChooser.showOpenDialog( /*AjpGUI.this*/ null );
	    	
	    	// If the user cancelled, then we just return.
	    	if( returnValue != JFileChooser.APPROVE_OPTION ) return;
	    	
	    	application.loadClass( fileChooser.getSelectedFile() );
	    }
	}
	
	/**
	 * The "Start" action class.
	 */
	class StartAction extends AbstractAction {

		/**
		 * The serial version UID.
		 */
		private static final long serialVersionUID = 1535729L;
		
		
		/**
		 * Creates a Start object, with its default values.
		 */
		public StartAction () {
			this( "Start", new ImageIcon( AjpFileTools.getImage( "start.png", AjpFileTools.IMAGE_TYPE_ICON ) ), "Executes the currently loaded class.", new Integer( 'A' ));
		}
		
		/**
		 * Creates a Start object.
		 * 
		 * @param text The text.
		 * @param icon The icon.
		 * @param desc The description.
		 * @param mnemonic The mnemonic.
		 */
	    public StartAction ( String text, ImageIcon icon,
	                       String desc, Integer mnemonic) {
	    	// We call the parent constructor.
	    	super(text, icon);
	    	// We set the description and the mnemonic.
	        putValue(SHORT_DESCRIPTION, desc);
	        putValue(MNEMONIC_KEY, mnemonic);
	    }
	    
	    /**
	     * This function is called when the Start action is performed.
	     */
	    public void actionPerformed(ActionEvent e) {
	    	application.start();
	    }
	}
	
	/**
	 * The "Stop" action class.
	 */
	class StopAction extends AbstractAction {

		/**
		 * The serial version UID.
		 */
		private static final long serialVersionUID = 1532329L;
		
		
		/**
		 * Creates a Start object, with its default values.
		 */
		public StopAction () {
			this( "Stop", new ImageIcon( AjpFileTools.getImage( "stop.png", AjpFileTools.IMAGE_TYPE_ICON ) ), "Stops the current execution.", new Integer( 'P' ));
		}
		
		/**
		 * Creates a Start object.
		 * 
		 * @param text The text.
		 * @param icon The icon.
		 * @param desc The description.
		 * @param mnemonic The mnemonic.
		 */
	    public StopAction ( String text, ImageIcon icon,
	                       String desc, Integer mnemonic) {
	    	// We call the parent constructor.
	    	super(text, icon);
	    	// We set the description and the mnemonic.
	        putValue(SHORT_DESCRIPTION, desc);
	        putValue(MNEMONIC_KEY, mnemonic);
	    }
	    
	    /**
	     * This function is called when the Start action is performed.
	     */
	    public void actionPerformed(ActionEvent e) {
	    	application.stop();
	    }
	}
	
	/**
	 * The "Quit" action class.
	 */
	class QuitAction extends AbstractAction {
		
		/**
		 * The serial version UID.
		 */
		private static final long serialVersionUID = 123452L;
		
		/**
		 * Creates a QuitAction object, with its default values.
		 */
		public QuitAction () {
			this("Quit", new ImageIcon( AjpFileTools.getImage( "quit.png", AjpFileTools.IMAGE_TYPE_ICON ) ), "Quits the application.", new Integer( 'Q' ));
		}
		
		/**
		 * Creates a QuitAction object.
		 * 
		 * @param text The text.
		 * @param icon The icon.
		 * @param desc The description.
		 * @param mnemonic The mnemonic.
		 */
	    public QuitAction ( String text, ImageIcon icon,
	                       String desc, Integer mnemonic) {
	    	// We call the parent constructor.
	    	super(text, icon);
	    	// We set the description and the mnemonic.
	        putValue(SHORT_DESCRIPTION, desc);
	        putValue(MNEMONIC_KEY, mnemonic);
	    }
	    
	    /**
	     * This function is called when the Quit action is performed.
	     */
	    public void actionPerformed(ActionEvent e) {
	    	// Exits the application
	        System.exit( 0 );
	    }
	}
	
	public final static URL getResource( final String filename )
	{
		// Try to load resource from jar
		URL url = ClassLoader.getSystemResource( filename );
		// If not found in jar, then load from disk
		if ( url == null ) {
			try {
				url = new URL( "file", "localhost", filename );
			} catch ( Exception urlException ) { } // ignore
		}
		return url;
	}
}