package org.lozi.ajp.ajp;

import static java.lang.Math.*;
import net.java.games.jogl.*;
import static net.java.games.jogl.GL.*;

import net.java.games.jogl.util.*;

import java.awt.event.*;
import java.awt.image.*;
import java.nio.*;
import java.util.ArrayList;

/**
 * AjpGLEventListener class.
 * 
 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
 * @version 1.0
 */
public class AjpGLEventListener implements GLEventListener, KeyListener
{
	/** The number of textures. */
	private static final int NUMBER_OF_TEXTURES = 8;
	
	/** The id of the blue texture. */
	public static final int BLUE_TEXTURE = 0;
	/** The file name of the blue texture. */
	public static final String BLUE_TEXTURE_FILE_NAME = "blue.png";
	/** The id of the cyan texture. */
	public static final int CYAN_TEXTURE = 1;
	/** The file name of the cyan texture. */
	public static final String CYAN_TEXTURE_FILE_NAME = "cyan.png";
	/** The green texture id . */
	public static final int GREEN_TEXTURE = 2;
	/** The file name of the green texture. */
	public static final String GREEN_TEXTURE_FILE_NAME = "green.png";
	/** The orange texture id. */
	public static final int ORANGE_TEXTURE = 3;
	/** The file name of the orange texture. */
	public static final String ORANGE_TEXTURE_FILE_NAME = "orange.png";
	/** The pink texture id. */
	public static final int PINK_TEXTURE = 4;
	/** The file name of the pink texture. */
	public static final String PINK_TEXTURE_FILE_NAME = "pink.png";
	/** The purple texture id. */
	public static final int PURPLE_TEXTURE = 5;
	/** The file name of the purple texture. */
	public static final String PURPLE_TEXTURE_FILE_NAME = "purple.png";
	/** The red texture id. */
	public static final int RED_TEXTURE = 6;
	/** The file name of the red texture. */
	public static final String RED_TEXTURE_FILE_NAME = "red.png";
	/** The yellow texture id. */
	public static final int YELLOW_TEXTURE = 7;
	/** The file name of the yellow texture. */
	public static final String YELLOW_TEXTURE_FILE_NAME = "yellow.png";
	
	/** This array contains the different textures. */
	private int[] texturesArray = new int[ NUMBER_OF_TEXTURES ];
	
	/** The left/right rotation step. */
	private double leftRightRotationStep;
	/** The top/bottom rotation step. */
	private double topBottomRotationStep;
	/** The zoom step. */
	private double zoom;
	/** The surfaces list. */
	private ArrayList<AjpUserSurface> surfacesArrayList;
	/** The list of the parametered curves. */
	private ArrayList<AjpUserParameteredCurve> parameteredCurvesArrayList;
	/** The list of the parametered surfaces. */
	private ArrayList<AjpUserParameteredSurface> parameteredSurfacesArrayList;
	/** The list of the parametered surfaces. */
	private ArrayList<AjpUserDot> dotsArrayList;
	/** The x lower bound. */
	private double xLowerBound;
	/** The x upper bound. */
	private double xUpperBound;
	/** The y lower bound. */
	private double yLowerBound;
	/** The y upper bound. */
	private double yUpperBound;
	/** The z lower bound. */
	private double zLowerBound;
	/** The z upper bound. */
	private double zUpperBound;
	/** The normals' lengths. */
	private double normalLength = 8;
	/** The background color. */
	private AjpUserColor backgroundColor;
	/** The axes' color. */
	private AjpUserColor axesColor;
	/** The left/right rotation. */
	private double leftRightRotation;
	/** The top/bottom rotation. */
	private double topBottomRotation;
	/** The lights' array. */
	private AjpUserLight[] lightsArray;
	/** The window. */
	private AjpUserWindow window;
	/** The window's action. */
	private AjpUserWindowAction action;
	/** The initial camera position. */
	private AjpUser3DPoint initialCameraPosition;
	/** The x initial rotation. */
	private double xInitialCameraRotation;
	/** The y initial rotation. */
	private double yInitialCameraRotation;
	/** The z initial rotation. */
	private double zInitialCameraRotation;
	
	/**
	 * Builds a new AjpGLEventListener.
	 * 
	 * @param lightsArray The lights.
	 */
	public AjpGLEventListener( AjpUserLight[] lightsArray ) {
		this.lightsArray = lightsArray;
	}
	
	/**
	 * Adds the given surface.
	 * 
	 * @param surface The surface to add.
	 */
	public synchronized void addSurface ( AjpUserSurface surface ) {
		surfacesArrayList.add( surface );
	}
	
	/**
	 * Adds the given parametric curve (one parameter).
	 * 
	 * @param curve The curve to add.
	 */
	public synchronized void addParameteredCurve ( AjpUserParameteredCurve curve ) {
		parameteredCurvesArrayList.add( curve );
	}
	
	/**
	 * Adds the given parametric surface (two parameters).
	 * 
	 * @param surface The surface to add.
	 */
	public synchronized void addParameteredSurface ( AjpUserParameteredSurface surface ) {
		parameteredSurfacesArrayList.add( surface );
	}
	
	/**
	 * Adds the given parametric dot.
	 * 
	 * @param dot The dot to add.
	 */
	public synchronized void addDot ( AjpUserDot dot ) {
		dotsArrayList.add( dot );
	}
	
	/**
	 * Sets the drawing bounds.
	 * 
	 * @param xLowerBound The x lower bound.
	 * @param xUpperBound The x upper bound.
	 * @param yLowerBound The y lower bound.
	 * @param yUpperBound The y upper bound.
	 * @param zLowerBound The z lower bound.
	 * @param zUpperBound The z upper bound.
	 */
	public void setBounds(	double xLowerBound, double xUpperBound, double yLowerBound, double yUpperBound, double zLowerBound,
			double zUpperBound ) {
		this.xLowerBound = xLowerBound;
		this.xUpperBound = xUpperBound;
		this.yLowerBound = yLowerBound;
		this.yUpperBound = yUpperBound;
		this.zLowerBound = zLowerBound;
		this.zUpperBound = zUpperBound;
	}
	
	/**
	 * @param backgroundColor The background color to set.
	 */
	public void setBackgroundColor(AjpUserColor backgroundColor) {
		this.backgroundColor = backgroundColor;
	}
	
	/**
	 * This function quickly creates a texture.
	 * 
	 * @param drawable The drawable.
	 * @param textureID The texture id.
	 * @param textureName The texture name.
	 */
	public void createTexture( GLDrawable drawable, int textureId, String textureName ) {
		drawable.getGL().glBindTexture( GL_TEXTURE_2D, texturesArray[ textureId ] );
		drawable.getGL().glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
		drawable.getGL().glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
		makeRGBTexture( drawable.getGL(), drawable.getGLU(), readPNGImage( textureName ), GL_TEXTURE_2D, true );
	}
	
	/**
	 * Initializes the event listener.
	 * 
	 * @param drawable The GLDrawable where to draw.
	 */
	public void init ( GLDrawable drawable )
	{
		// We initialize the surfaces array.
		surfacesArrayList = new ArrayList<AjpUserSurface>();
		parameteredCurvesArrayList = new ArrayList<AjpUserParameteredCurve>();
		parameteredSurfacesArrayList = new ArrayList<AjpUserParameteredSurface>();
		dotsArrayList = new ArrayList<AjpUserDot>();
		
		if( initialCameraPosition == null ) initialCameraPosition = new AjpUser3DPoint( 0, 0, -20 );
		xInitialCameraRotation = -65;
		yInitialCameraRotation = 0;
		zInitialCameraRotation = -45;
		if( backgroundColor == null ) backgroundColor = new AjpUserColor( 1.0f, 1.0f, 1.0f, 1.0f );
		// We first get the GL and GLU objects.
		final GL gl = drawable.getGL();
		final GLU glu = drawable.getGLU();
		
		// We need to listen to the key events.
		drawable.addKeyListener(this);
		
		// We set miscellaneous properties.
		gl.glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
		gl.glShadeModel( GL_SMOOTH );
		gl.glClearDepth( 1.0f );
		gl.glEnable( GL_DEPTH_TEST );
		gl.glDepthFunc( GL_LEQUAL );
		gl.glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );
		
		
		int i = 0;
		
		for( AjpUserLight light : lightsArray ) {
			// No more than 7 lights!
			if( i == 7 ) break;
			
			// Lighting
			float LightAmbient[] = { 1f, 1f, 1f, 1.0f };
			float LightDiffuse[] = { light.getColor().getRed(), light.getColor().getGreen(), light.getColor().getBlue(), light.getColor().getAlpha() };
			float LightPosition[] = { (float)light.getPosition().getX(), (float)light.getPosition().getY(), (float)light.getPosition().getZ(), 1.0f };
			
			int lightNumber;
			// Well the values provided by OpenGL are not in an array, so we have no choice.
			switch( i ) {
			case 0 : lightNumber = GL_LIGHT1; break;
			case 1 : lightNumber = GL_LIGHT2; break;
			case 2 : lightNumber = GL_LIGHT3; break;
			case 3 : lightNumber = GL_LIGHT4; break;
			case 4 : lightNumber = GL_LIGHT5; break;
			case 5 : lightNumber = GL_LIGHT6; break;
			case 6 : lightNumber = GL_LIGHT7; break;
			case 7 : lightNumber = GL_LIGHT0; break;
			default : lightNumber = GL_LIGHT1; break;
			}
			gl.glLightfv( lightNumber, GL_AMBIENT, LightAmbient );
			gl.glLightfv( lightNumber, GL_DIFFUSE, LightDiffuse );
			gl.glLightfv( lightNumber, GL_POSITION, LightPosition );
			gl.glEnable( lightNumber );
			
			i++;
		}
		
		gl.glEnable( GL_LIGHTING );
		gl.glEnable( GL_TEXTURE_2D );
		
		// Now we create the different textures.
		gl.glGenTextures( NUMBER_OF_TEXTURES, this.texturesArray );
		// One for each color :)
		createTexture( drawable, BLUE_TEXTURE, BLUE_TEXTURE_FILE_NAME );
		createTexture( drawable, CYAN_TEXTURE, CYAN_TEXTURE_FILE_NAME );
		createTexture( drawable, GREEN_TEXTURE, GREEN_TEXTURE_FILE_NAME );
		createTexture( drawable, ORANGE_TEXTURE, ORANGE_TEXTURE_FILE_NAME );
		createTexture( drawable, PINK_TEXTURE, PINK_TEXTURE_FILE_NAME );
		createTexture( drawable, PURPLE_TEXTURE, PURPLE_TEXTURE_FILE_NAME );
		createTexture( drawable, RED_TEXTURE, RED_TEXTURE_FILE_NAME );
		createTexture( drawable, YELLOW_TEXTURE, YELLOW_TEXTURE_FILE_NAME );
		
		
		//gl.glColor4f(.5f,0.5f,0.5f,0.9f);// Full Brightness, 50% Alpha ( NEW )
		//gl.glBlendFunc(GL_SRC_ALPHA,GL_ONE);// Blending Function For Translucency Based On Source Alpha Value ( NEW )
		
		//gl.glEnable(GL_BLEND);// Turn Blending On
		//gl.glDisable(GL_DEPTH_TEST);
	}
	
	/**
	 * Converts a color string to a texture.
	 * 
	 * @param color The color string to consider.
	 * @return Its integer value.
	 */
	public int colorStringToTexture ( String color ) {
		if( color.toLowerCase().equals("blue") ) return BLUE_TEXTURE;
		else if( color.toLowerCase().equals("cyan") ) return CYAN_TEXTURE;
		else if( color.toLowerCase().equals("green") ) return GREEN_TEXTURE;
		else if( color.toLowerCase().equals("orange") ) return ORANGE_TEXTURE;
		else if( color.toLowerCase().equals("pink") ) return PINK_TEXTURE;
		else if( color.toLowerCase().equals("purple") ) return PURPLE_TEXTURE;
		else if( color.toLowerCase().equals("red") ) return RED_TEXTURE;
		else if( color.toLowerCase().equals("yellow") ) return YELLOW_TEXTURE;
		else return 0;
	}
	
	/**
	 * This function is called each time the drawable is drawn. :)
	 * 
	 * @param drawable The GLDrawable where to draw.
	 */
	public synchronized void display ( GLDrawable drawable )
	{
		// We run the action each time the frame is refreshed.
		if( action != null )
			
			action.run( window );
		
		// We get the gl object.
		final GL gl = drawable.getGL();
		
		if ( axesColor == null ) axesColor = new AjpUserColor( 1.0f, 1.0f, 1.0f, 1.0f );
		
		gl.glClearColor( backgroundColor.getRed(), backgroundColor.getGreen(), backgroundColor.getBlue(), backgroundColor.getAlpha() );
		
		// We clear the drawing area.
		gl.glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
		// We start from the identity.
		gl.glLoadIdentity();
		
		// We update the camera.
		
		gl.glTranslated( initialCameraPosition.getX(), initialCameraPosition.getY(), initialCameraPosition.getZ() + (float)zoom );
		gl.glRotated( xInitialCameraRotation,1.0, 0.0, 0.0 );
		gl.glRotated( yInitialCameraRotation,0.0, 1.0, 0.0 );
		gl.glRotated( zInitialCameraRotation,0.0, 0.0, 1.0 );
		
		gl.glRotated(leftRightRotation,0.0,0.0,1.0);
		gl.glRotated(topBottomRotation,1.0,0.0,0.0);
		
		// We want light!
		gl.glEnable( GL_LIGHTING );
		
		// For each surface...
		for( AjpUserSurface surface : surfacesArrayList ) {
			// We initialize the surface.
			surface.init();
			
			// Those variable will represent the current function's value for each
			// vertex of the square.
			double fBottomLeft, fTopLeft, fBottomRight, fTopRight;
			
			// We get the grid size in the h variable for more lisibility.
			double h = surface.getGridSize();
			
			// We bind the right texture to this surface.
			gl.glBindTexture( GL_TEXTURE_2D, texturesArray[colorStringToTexture( surface.getColorString() )]);
			
			// We are going to draw the surface, using triangles.
			gl.glBegin( GL_TRIANGLES );
			
			// Now we are going to divide the (Ox,Oy) axis into 
			// squares and to draw the surfaces using triangles.
			// We start from the x lower bound, and step by the grid size.
			for ( double x = surface.getXLowerBound() ; x <= surface.getXUpperBound() ; x += surface.getGridSize() ) {
				// We start from the y lower bound, and step by the grid size.
				for ( double y = surface.getYLowerBound() ; y <= surface.getYUpperBound() ; y += surface.getGridSize() ) {
					// 
					// Seen from above :  [^y >x]
					//  
					//  fTopLeft    +---+ fTopRight
					//              | \ |
					//  fBottomLeft +---+ fBottomRight
					//  
					
					// We compute the values of the 4 first points ( the lower square ).
					fBottomLeft = surface.evaluate( x, y );
					fTopLeft = surface.evaluate( x + h, y );
					fBottomRight = surface.evaluate( x, y + h );
					fTopRight = surface.evaluate( x + h, y + h );
					
					// We compute the dot product of the two first edges of the "square".
					gl.glNormal3d(	-h * ( fTopLeft - fBottomLeft ) * normalLength,
							-h * ( fBottomRight - fBottomLeft ) * normalLength,
							h * h * normalLength );
					
					// Then we draw the first triangle.
					gl.glTexCoord2d( 0.0, 0.0 ); gl.glVertex3d( x, y, fBottomLeft );
					gl.glTexCoord2d( 1.0, 0.0 ); gl.glVertex3d( x + h, y, fTopLeft );
					gl.glTexCoord2d( 0.0, 1.0 ); gl.glVertex3d( x, y + h, fBottomRight );
					
					// We compute the dot product of the two remaining edges of the "square".
					gl.glNormal3d(	-h * ( fTopRight - fBottomRight ) * normalLength,
							-h * ( fTopRight - fTopLeft ) * normalLength,
							h * h * normalLength );
					
					// Then we draw the second triangle.
					gl.glTexCoord2d( 1.0, 0.0 ); gl.glVertex3d( x + h, y, fTopLeft );
					gl.glTexCoord2d( 1.0, 1.0 ); gl.glVertex3d( x + h, y + h, fTopRight );
					gl.glTexCoord2d( 0.0, 1.0 ); gl.glVertex3d( x, y + h, fBottomRight );
				}
			}
			
			// We finished drawing the surface.
			gl.glEnd();
		}
		
		// For each parametric surface...
		for( AjpUserParameteredSurface surface : parameteredSurfacesArrayList ) {
			// We initialize the surface.
			surface.init();
			
			// We bind the right texture to this surface.
			gl.glBindTexture( GL_TEXTURE_2D, texturesArray[colorStringToTexture( surface.getColorString() )]);
			
			// We are going to draw the surface, using triangles.
			gl.glBegin( GL_TRIANGLES );
			
			// We get the two steps.
			double sStep = surface.getSStep();
			double tStep = surface.getTStep();
			
			// For each value of the s parameter...
			for ( double s = surface.getSLowerBound() ; s <= surface.getSUpperBound() ; s += surface.getSStep() ) {
				// ...and for each value of the t parameter...
				for( double t = surface.getTLowerBound() ; t <= surface.getTUpperBound() ; t += surface.getTStep() ) {
					// ...we draw two triangles.
					// We get the four points
					AjpUser3DPoint fBottomLeft = surface.evaluate( s, t );
					AjpUser3DPoint fTopLeft = surface.evaluate( s + sStep, t );
					AjpUser3DPoint fBottomRight = surface.evaluate( s, t + tStep );
					AjpUser3DPoint fTopRight = surface.evaluate( s + sStep, t + tStep );
					
					// v ( x1, y1, z1 )
					double x1 = fBottomRight.getX() - fBottomLeft.getX();
					double y1 = fBottomRight.getY() - fBottomLeft.getY();
					double z1 = fBottomRight.getZ() - fBottomLeft.getZ();
					
					// w ( x2, y2, z2 )
					double x2 = fTopLeft.getX() - fBottomRight.getY();
					double y2 = fTopLeft.getY() - fBottomRight.getY();
					double z2 = fTopLeft.getZ() - fBottomRight.getZ();
					
					// We compute the dot product of the two first edges of the "square".
					gl.glNormal3d(	( y1 * z2 - y2 * z1 ) * normalLength,
							( z1 * x2 - x1 * z2 ) * normalLength,
							( x1 * y2 - x1 * y1 ) * normalLength );
					
					// Then we draw the three points.
					gl.glTexCoord2d( 0.0, 0.0 ); gl.glVertex3d( fBottomLeft.getX(), fBottomLeft.getY(), fBottomLeft.getZ() );
					gl.glTexCoord2d( 1.0, 0.0 ); gl.glVertex3d( fTopLeft.getX(), fTopLeft.getY(), fTopLeft.getZ() );
					gl.glTexCoord2d( 0.0, 1.0 ); gl.glVertex3d( fBottomRight.getX(), fBottomRight.getY(), fBottomRight.getZ() );
					
					// v ( x1, y1, z1 )
					x1 = fTopLeft.getX() - fTopRight.getX();
					y1 = fTopLeft.getY() - fTopRight.getY();
					z1 = fTopLeft.getZ() - fTopRight.getZ();
					
					// w ( x2, y2, z2 )
					x2 = fBottomRight.getX() - fTopRight.getY();
					y2 = fBottomRight.getY() - fTopRight.getY();
					z2 = fBottomRight.getZ() - fTopRight.getZ();
					
					// We compute the dot product of the two remaning edges of the "square".
					gl.glNormal3d(	( y1 * z2 - y2 * z1 ) * normalLength,
							( z1 * x2 - x1 * z2 ) * normalLength,
							( x1 * y2 - x1 * y1 ) * normalLength );
					
					// Then we draw the three points.
					gl.glTexCoord2d( 1.0, 0.0 ); gl.glVertex3d( fTopLeft.getX(), fTopLeft.getY(), fTopLeft.getZ() );
					gl.glTexCoord2d( 1.0, 1.0 ); gl.glVertex3d( fTopRight.getX(), fTopRight.getY(), fTopRight.getZ() );
					gl.glTexCoord2d( 0.0, 1.0 ); gl.glVertex3d( fBottomRight.getX(), fBottomRight.getY(), fBottomRight.getZ() );
					
				}
			}
		}
		
		// We don't want lighting for the curves.
		gl.glDisable( GL_LIGHTING );
		
		// For each parametric ( 1 parameter ) curve...
		for( AjpUserParameteredCurve curve : parameteredCurvesArrayList ) {
			// We initialize the surface.
			curve.init();
			// We set the drawing color.
			gl.glColor3f( curve.getColor().getRed(), curve.getColor().getGreen(), curve.getColor().getBlue() );
			// We first get the drawing step.
			double h = curve.getStep();
			// Then we set the line width.
			gl.glLineWidth( curve.getLineWidth() );
			// We are going to draw lines.
			gl.glBegin( GL_LINE_STRIP );
			// Now we step from a value to the next.
			for( double t = curve.getTLowerBound() ; t < curve.getTUpperBound() ; t += h ) {
				// We want a line which starts from this point...
				gl.glVertex3d( curve.evaluate( t ).getX(), curve.evaluate( t ).getY(), curve.evaluate( t ).getZ() );
				// ...to the next.
				gl.glVertex3d( curve.evaluate( t + h ).getX(), curve.evaluate( t + h ).getY(), curve.evaluate( t + h ).getZ() );
			}
			// Done!
			gl.glEnd();
		}
		
		// For each dot...
		for( AjpUserDot dot : dotsArrayList ) {
			// We initialize the dot.
			dot.init();
			// We set the drawing color.
			gl.glColor3f( dot.getColor().getRed(), dot.getColor().getGreen(), dot.getColor().getBlue() );
			// We are going to draw points.
			gl.glBegin( GL_POINTS );
			// Let's plot :)
			gl.glVertex3d( dot.getCoordinates().getX(), dot.getCoordinates().getY(), dot.getCoordinates().getZ() );
			// Done!
			gl.glEnd();
		}
		
		// Now we draw the axes.
		gl.glColor3f( axesColor.getRed(), axesColor.getGreen(), axesColor.getBlue() );
		
		// We want a thick line.
		gl.glLineWidth( 1.5f );
		// Then we draw the axes.
		gl.glBegin( GL_LINES );
		gl.glVertex3d( 0.0, 0.0, zLowerBound );
		gl.glVertex3d( 0.0, 0.0, zUpperBound );
		gl.glVertex3d( 0.0, yLowerBound * 1.3, 0.0 );
		gl.glVertex3d( 0.0, yUpperBound * 1.3, 0.0 );
		gl.glVertex3d( xLowerBound * 1.3, 0.0, 0.0 );
		gl.glVertex3d( xUpperBound * 1.3, 0.0, 0.0 );
		gl.glEnd();
		
		// Then we flush the buffer.
		gl.glFlush(); 
		
		leftRightRotation += leftRightRotationStep;
		topBottomRotation += topBottomRotationStep;
	}
	
	/**
	 * This function is called when the window size changes.
	 * 
	 * @param drawable The drawable.
	 * @param i Ignored.
	 * @param x Ignored.
	 * @param width The new window's width.
	 * @param height The new window"s height.
	 */
	public void reshape ( GLDrawable drawable, int i, int x, int width, int height )
	{
		final GL gl = drawable.getGL();
		final GLU glu = drawable.getGLU();
		
		if ( height <= 0 )  // To avoid a divide by zero error.
			height = 1;
		final float h = ( float ) width / ( float ) height;
		gl.glViewport( 0, 0, width, height );
		gl.glMatrixMode( GL_PROJECTION );
		gl.glLoadIdentity();
		glu.gluPerspective( 85.0f, h, 1.0, 220.0/*Float.MAX_VALUE*/ );
		gl.glMatrixMode( GL_MODELVIEW );
		gl.glLoadIdentity();
		gl.glTranslatef( 0.0f, 0.0f, 5.0f );
	}
	
	/**
	 * This function is called when the display has changed.
	 */
	public void displayChanged ( GLDrawable drawable, boolean modeChanged, boolean deviceChanged ) { }
	
	/**
	 * This function handles the key released events.
	 * 
	 * @param e The event to handle.
	 */
	public void keyReleased( KeyEvent e ) {}
	
	/**
	 * This function handles the key pressed events.
	 * 
	 * @param e The event to handle.
	 */
	public void keyPressed( KeyEvent e )
	{
		switch( e.getKeyCode() ) {
		// These are the default actions : rotations and zoom.
		case KeyEvent.VK_LEFT :
			leftRightRotation-=0.5;
			break;
		case KeyEvent.VK_RIGHT :
			leftRightRotation+=0.5;
			break;
		case KeyEvent.VK_UP :
			topBottomRotation-=0.5;
			break;
		case KeyEvent.VK_DOWN :
			topBottomRotation+=0.5;
			break;
		case KeyEvent.VK_PAGE_UP:
			zoom+=0.3;
			break;
		case KeyEvent.VK_PAGE_DOWN:
			zoom-=0.3;
			break;
		}
	}
	
	/**
	 * This function handles the key typed events.
	 * 
	 * @param e The event to handle.
	 */
	public void keyTyped(KeyEvent e) {}
	
	/**
	 * Reads a PNG image from its ressource name.
	 * 
	 * @param resourceName The ressource name.
	 * @return The buffered image.
	 */
	private BufferedImage readPNGImage( String resourceName )
	{
		// We first get the image.
		BufferedImage img = AjpFileTools.getImage( resourceName, AjpFileTools.IMAGE_TYPE_TEXTURE );
		// Then we scale it...
		java.awt.geom.AffineTransform tx = java.awt.geom.AffineTransform.getScaleInstance( 1, -1 );
		// ...and translate it.
		tx.translate( 0, -img.getHeight( null ) );
		AffineTransformOp op = new AffineTransformOp( tx, AffineTransformOp.TYPE_NEAREST_NEIGHBOR );
		img = op.filter( img, null );
		// Then we return the image.
		return img;
	}
	
	/**
	 * Makes an RGB texture.
	 * 
	 * @param gl The GL context.
	 * @param glu The GLU.
	 * @param img The image to start from.
	 * @param target The target in the textures' array.
	 * @param mipmapped Should we mipmap the texture?
	 */
	private void makeRGBTexture( GL gl, GLU glu, BufferedImage img, int target, boolean mipmapped )
	{
		// We initialize the byte buffer.
		ByteBuffer dest = null;
		// Then, we create the texture, depending on the image type.
		switch ( img.getType() ) {
		case BufferedImage.TYPE_3BYTE_BGR:
		case BufferedImage.TYPE_CUSTOM: {
			byte[] data = ( ( DataBufferByte ) img.getRaster().getDataBuffer() ).getData();
			dest = ByteBuffer.allocateDirect( data.length );
			dest.order( ByteOrder.nativeOrder() );
			dest.put( data, 0, data.length );
			break;
		}
		case BufferedImage.TYPE_INT_RGB: {
			int[] data = ( ( DataBufferInt ) img.getRaster().getDataBuffer() ).getData();
			dest = ByteBuffer.allocateDirect( data.length * BufferUtils.SIZEOF_INT );
			dest.order( ByteOrder.nativeOrder() );
			dest.asIntBuffer().put( data, 0, data.length );
			break;
		}
		default:
			throw new RuntimeException( "Unsupported image type " + img.getType() );
		}
		
		if ( mipmapped ) {
			glu.gluBuild2DMipmaps( target, GL.GL_RGB8, img.getWidth(), img.getHeight(), GL.GL_RGB, GL.GL_UNSIGNED_BYTE, dest );
		} else {
			gl.glTexImage2D( target, 0, GL.GL_RGB, img.getWidth(), img.getHeight(), 0, GL.GL_RGB, GL.GL_UNSIGNED_BYTE, dest );
		}
	}
	
	/**
	 * Sets the initial camera position and direction.
	 * 
	 * @param Ajp3DUserPoint position The camera position.
	 * @param xRot The x initial rotation.
	 * @param yDir The y initial rotation.
	 * @param zDir The z initial rotation.
	 */
	public void setInitialCameraPositionAndDirection( AjpUser3DPoint position, double xRot, double yRot, double zRot ) {
		this.initialCameraPosition = position;
		this.xInitialCameraRotation = xRot;
		this.yInitialCameraRotation = yRot;
		this.zInitialCameraRotation = zRot;
	}
	
	/**
	 * @param leftRightRotationStep The leftRightRotationStep to set.
	 */
	public void setLeftRightRotationStep(double leftRightRotationStep) {
		this.leftRightRotationStep = leftRightRotationStep;
	}
	
	/**
	 * @param topBottomRotationStep The topBottomRotationStep to set.
	 */
	public void setTopBottomRotationStep(double topBottomRotationStep) {
		this.topBottomRotationStep = topBottomRotationStep;
	}
	
	/**
	 * @param lightsArray The lightsArray to set.
	 */
	public synchronized void setLightsArray(AjpUserLight[] lightsArray) {
		this.lightsArray = lightsArray;
	}
	
	/**
	 * @return Returns the parameteredCurvesArrayList.
	 */
	public synchronized ArrayList<AjpUserParameteredCurve> getParameteredCurvesArrayList() {
		return parameteredCurvesArrayList;
	}
	
	/**
	 * @param parameteredCurvesArrayList The parameteredCurvesArrayList to set.
	 */
	public synchronized void setParameteredCurvesArrayList(
			ArrayList<AjpUserParameteredCurve> parameteredCurvesArrayList) {
		this.parameteredCurvesArrayList = parameteredCurvesArrayList;
	}
	
	/**
	 * @return Returns the parameteredSurfacesArrayList.
	 */
	public synchronized ArrayList<AjpUserParameteredSurface> getParameteredSurfacesArrayList() {
		return parameteredSurfacesArrayList;
	}
	
	/**
	 * @param parameteredSurfacesArrayList The parameteredSurfacesArrayList to set.
	 */
	public synchronized void setParameteredSurfacesArrayList(
			ArrayList<AjpUserParameteredSurface> parameteredSurfacesArrayList) {
		this.parameteredSurfacesArrayList = parameteredSurfacesArrayList;
	}
	
	/**
	 * @return Returns the surfacesArrayList.
	 */
	public synchronized ArrayList<AjpUserSurface> getSurfacesArrayList() {
		return surfacesArrayList;
	}
	
	/**
	 * @param surfacesArrayList The surfacesArrayList to set.
	 */
	public synchronized void setSurfacesArrayList(ArrayList<AjpUserSurface> surfacesArrayList) {
		this.surfacesArrayList = surfacesArrayList;
	}
	
	/**
	 * @return Returns the dotsArrayList.
	 */
	public synchronized ArrayList<AjpUserDot> getDotsArrayList() {
		return dotsArrayList;
	}
	
	/**
	 * @param dotsArrayList The dotsArrayList to set.
	 */
	public synchronized void setDotsArrayList(ArrayList<AjpUserDot> dotsArrayList) {
		this.dotsArrayList = dotsArrayList;
	}
	
	/**
	 * @return Returns the axes' color.
	 */
	public AjpUserColor getAxesColor() {
		return axesColor;
	}
	
	/**
	 * @param axesColor The axes' color to set.
	 */
	public void setAxesColor(AjpUserColor axesColor) {
		this.axesColor = axesColor;
	}
	
	/**
	 * @param action The action to set.
	 */
	public void setAction(AjpUserWindowAction action) {
		this.action = action;
	}
	
	/**
	 * @param window The window to set.
	 */
	public void setWindow(AjpUserWindow window) {
		this.window = window;
	}
}
