package org.lozi.ajp.ajp;

import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import javax.imageio.ImageIO;

/**
 * AjpFileTools class. Contains miscellaneous tools for the Ajp project.
 * 
 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
 * @version 1.0
 */
public class AjpFileTools {
	/** The OS dependant file separator. */
	public static final String FILE_SEPARATOR = java.io.File.separator;
	/** The package's path. */
	public static final String PACKAGE_PATH = "org" + FILE_SEPARATOR + "lozi" + FILE_SEPARATOR
												+ "ajp" + FILE_SEPARATOR + "ajp";
	/** The name of the folder containing the scripts. */
	public static final String SCRIPTS_FOLDER_NAME = "scripts";
	/** The name of the folder containing the classes. */
	public static final String CLASSES_FOLDER_NAME = "classes";
	/** The name of the folder containing the sources. */
	public static final String SOURCES_FOLDER_NAME = "sources";
	/** The name of the folder containing the images. */
	public static final String IMAGES_FOLDER_NAME = "images";
	/** The name of the folder containing the textures. */
	public static final String TEXTURES_FOLDER_NAME = "textures";
	/** The name of the folder containing the icons. */
	public static final String ICONS_FOLDER_NAME = "icons";
	/** The "texture" image type. */
	public static final int IMAGE_TYPE_TEXTURE = 0;
	/** The "icon" image type. */
	public static final int IMAGE_TYPE_ICON = 1;
	
	/**
	 * Gets an image from its name. Automatically resolves the image file,
	 * and returns the Image object.
	 * 
	 * @param name The image name.
	 * @param type The image type.
	 * @return The BufferedImage object, null if it could not be found.
	 */
	public static BufferedImage getImage( String name, int type ) {
		// The images folder is in the classes folder's parent.
		URL imageURL;
		try {
			// The images folder is in the classes folder's parent.
			imageURL = new URL( "file:" + baseFolder().getPath() + FILE_SEPARATOR + IMAGES_FOLDER_NAME
						+ FILE_SEPARATOR + ( type == IMAGE_TYPE_TEXTURE ? TEXTURES_FOLDER_NAME : ICONS_FOLDER_NAME )
						+ FILE_SEPARATOR + name );
			
			// We create the image and return it.
			return ImageIO.read( imageURL );
		} catch (Exception e) {
			// We return null if we could not locate the file.
			return null;
		}
		
	}
	
	/**
	 * Returns the "sources" folder.
	 * 
	 * @return The "sources" folder.
	 */
	public static File sourcesFolder() {
		// We first get the classes' base URL.
		URL classesBaseURL = ClassLoader.getSystemResource("");
		// Then we return the File object of the "sources" folder.
		return new File( classesBaseURL.getPath().replaceAll("%20"," ").substring( 0, classesBaseURL.getPath().replaceAll("%20"," ").length() - CLASSES_FOLDER_NAME.length() - 1 )
						+ FILE_SEPARATOR + SOURCES_FOLDER_NAME );
	}
	
	/**
	 * Returns the "scripts" folder within the "sources" folder.
	 * 
	 * @return The "scripts" folder within the "sources" folder.
	 */
	public static File sourcesScriptFolder() {
		return new File( sourcesFolder().getPath() + FILE_SEPARATOR + PACKAGE_PATH + FILE_SEPARATOR + SCRIPTS_FOLDER_NAME );
	}
	
	/**
	 * Returns the "classes" folder.
	 * 
	 * @return The "classes" folder.
	 */
	public static File classesFolder() {
		// We first get the classes' base URL.
		URL classesBaseURL = ClassLoader.getSystemResource("");
		// Then we return the File object of the "classes" folder.
		return new File( classesBaseURL.getPath().replaceAll("%20"," ").substring( 0, classesBaseURL.getPath().replaceAll("%20"," ").length() - CLASSES_FOLDER_NAME.length() - 1 )
						+ FILE_SEPARATOR + CLASSES_FOLDER_NAME );
	}
	
	/**
	 * Returns the "scripts" folder within the "classes" folder.
	 * 
	 * @return The "scripts" folder within the "classes" folder.
	 */
	public static File classesScriptFolder() {
		return new File( classesFolder().getPath() + FILE_SEPARATOR + SCRIPTS_FOLDER_NAME );
	}
	
	/**
	 * Returns the "ajp" folder.
	 * 
	 * @return The "ajp" folder.
	 */
	public static File baseFolder() {
		// We first get the classes' base URL.
		URL classesBaseURL = ClassLoader.getSystemResource("");
		
		// Then we return the File object of the "ajp" folder.
		return new File( classesBaseURL.getPath().replaceAll("%20"," ").substring( 0, classesBaseURL.getPath().replaceAll("%20"," ").length() - CLASSES_FOLDER_NAME.length() - 1 ) );
	}
	
	/**
	 * Returns the file which name is given as a parameter, assuming that the file
	 * belongs to the "sources" folder.
	 *  
	 * @param fileName The file name.
	 * @return The file object corresponding to the file.
	 */
	public static File fileInTheSourcesFolder( String fileName ) {
		// Then we return the File object of the file whose name was given as a parameter.
		
		return new File( sourcesFolder().getPath() + fileName );
	}
	
	/**
	 * Copies a file from a destination to another.
	 * 
	 * @param source The file to copy.
	 * @param destination The destination file.
	 * @throws IOException
	 */
	public static void fileCopy( File source, File destination ) throws IOException {
		
		// We first create the reader and the writer.
		BufferedReader reader = new BufferedReader( new FileReader( source ) );
		BufferedWriter writer = new BufferedWriter( new FileWriter( destination ) );
		
		// We intialize the buffer used to copy the file.
		char[] buffer = new char[32768];
		
		int read;
		
		// We simply copy the file.
		while( ( read = reader.read( buffer ) ) != -1 )
			writer.write( buffer, 0, read );
		
		// Then we close the buffers.
		reader.close();
		writer.close();
	}
	
	/**
	 * Cleans up the scripts' folders in the project path.
	 * This function should be called when the application starts and when it exits.
	 */
	public static void cleanUpTheScriptsFolders() {
		
		// We get the sources' scripts folder.
		File sourcesScriptsFolder = new File(	sourcesFolder().getPath()
												+ FILE_SEPARATOR + PACKAGE_PATH + FILE_SEPARATOR + SCRIPTS_FOLDER_NAME );
		// We get the classes' scripts folder.
		File classesScriptsFolder = new File(	classesFolder().getPath()
												+ FILE_SEPARATOR + PACKAGE_PATH + FILE_SEPARATOR + SCRIPTS_FOLDER_NAME );
		
		if( !sourcesScriptsFolder.isDirectory() ) AjpMiscellaneousTools.error( "Unable to find the scripts folder within the sources directory." );
		if( !classesScriptsFolder.isDirectory() ) AjpMiscellaneousTools.error( "Unable to find the scripts folder within the classes directory." );
		
		if( sourcesScriptsFolder.listFiles() != null ) {
			// We delete each file in the first folder.
			for( File currentFile : sourcesScriptsFolder.listFiles() )
				if( !currentFile.getName().equals("AjpUserScript.java") && !currentFile.getName().equals("AjpScriptLoader.java") )
					currentFile.delete();
		}
		
		if( classesScriptsFolder.listFiles() != null ) {
			// We delete each file in the second folder.
			for( File currentFile : classesScriptsFolder.listFiles() )
				if( !currentFile.getName().equals("AjpUserScript.class") && !currentFile.getName().equals("AjpScriptLoader.class") )
					currentFile.delete();
		}
		
	}
}