package org.lozi.ajp.ajp;

import org.lozi.ajp.ajp.scripts.*;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import javax.swing.JFrame;

/**
 * AjpApp class.
 * 
 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
 * @version 1.0
 */
public class AjpApp {
	/** The current script to execute. */
	private AjpUserScript script;
	/** The thread where the script is executed. */
	private Thread executionThread;
	/** The plotter that we will pass to the user. */
	private AjpUserPlotter plotter;
	/** The frame list. */
	private static ArrayList<AjpUserWindow> windowList;
	
	/**
	 * Returns an instance of the application.
	 */
	public AjpApp() {
		// We first clean up the two scripts folders.
		AjpFileTools.cleanUpTheScriptsFolders();
		// We create the GUI.
		AjpGUI gui = new AjpGUI();
		// The GU needs a pointer to the application in order to
		// call the application functions.
		gui.init( this );
		// Now we initialize the thread.
		executionThread = new Thread( new RunnableExecution() );
	
		windowList = new ArrayList<AjpUserWindow>();
		
		plotter = new AjpUserPlotter();
		
		// Before we exit...
		Runtime.getRuntime().addShutdownHook(new Thread () {
			public void run() {
				// ...we clean up the scripts folders.
				AjpFileTools.cleanUpTheScriptsFolders();
			}
		});
	}
	
	/**
	 * Returns the script.
	 * 
	 * @return Returns the script.
	 */
	public AjpUserScript getScript() {
		return script;
	}

	/**
	 * Sets the script.
	 * 
	 * @param script The script to set.
	 */
	public void setScript( AjpUserScript script ) {
		this.script = script;
	}
	
	/**
	 * Loads the given class.
	 * 
	 * @param classFile The class to load.
	 */
	public void loadClass( File classFile ) {
		// The file where the class will be copied.
		File newClassFile = new File(	AjpFileTools.classesScriptFolder()
										+ AjpFileTools.FILE_SEPARATOR + classFile.getName()  );
		
		try {
			// We try to copy the file to the scripts folder.
			AjpFileTools.fileCopy(	classFile, newClassFile );
		}
		catch( IOException ex ) {
			// If we could not copy the file we can not continue.
			AjpMiscellaneousTools.error( "Unable to copy the class file. Please check that you have the writing rights the scripts folder." );
			return;
		}
		
		loadClassFile( newClassFile );
	}
	
	/**
	 * Loads the given source file.
	 * 
	 * @param sourceFile The source file to load.
	 */
	public void loadSource( File sourceFile ) {
		
		// The file where the source will be copied.
		File newSourceFile = new File(	AjpFileTools.sourcesScriptFolder()
										+ AjpFileTools.FILE_SEPARATOR + sourceFile.getName() );
		
		try {
			// We try to copy the file to the scripts folder.
			AjpFileTools.fileCopy(	sourceFile, newSourceFile );
		}
		catch( IOException ex ) {
			// If we could not copy the file we can not continue.
			AjpMiscellaneousTools.error( "Unable to copy the source file. Please check that you have the writing rights the scripts folder." );
			return;
		}
		
		Process compilation = null;
		
		try {
			
			System.out.flush();
			// We try compile the source file.
			compilation = Runtime.getRuntime().exec( new String[] { "javac",  "-d",  AjpFileTools.classesFolder().getPath(), "-classpath", AjpFileTools.classesFolder().getPath(),
					 AjpFileTools.sourcesScriptFolder().toString() + AjpFileTools.FILE_SEPARATOR + newSourceFile.getName().toString() } , null,
										new File( AjpFileTools.baseFolder().getPath() ) );
			
			// We try to get the compilation errors.
			try {	
				BufferedReader br = new BufferedReader( new InputStreamReader( compilation.getErrorStream() ) );
				
				String line = null;
	            String errors = "";
	            
	            while ( (line = br.readLine()) != null)
	                errors += line + "\n";
				
				// Now we display the compilation errors (if any).
				if( errors.toString().length() > 1) {
					System.out.println("#### COMPILATION ERRORS ####\n" + errors );
					AjpMiscellaneousTools.error( "Unable to compile the source file!\n\n" + errors );
				}	

			} catch ( Exception ex ) {
				// If we could not get the compilation error, we display a simple message.
				AjpMiscellaneousTools.error( "Unable to get the compilation errors." );
			}
			
			// We wait for the command to end.
			compilation.waitFor();
			
		} catch (Exception e) {
			// If we could not compile the file we can not continue.
			AjpMiscellaneousTools.error( "Unable to compile the source file!" );
			
			return;
		}
		
		// We try to get the compilation errors.
		try {
			StringBuffer errorStr = new StringBuffer( "" );
			
			// If the compilation object was initialized...
			if( compilation != null ) {
				byte[] buffer = new byte[1024];
				
				// We read the error stream.
				while( compilation.getErrorStream().read( buffer ) != -1 )
				{
					errorStr.append( buffer );
				}
			}
			
			// Now we display the compilation errors (if any).
			if( errorStr.toString().length() > 1) {
				System.out.println("##" + errorStr );
				AjpMiscellaneousTools.error( "Unable to compile the source file!\n\n" + errorStr );
			}	
		} catch ( Exception ex ) {
			// If we could not get the compilation error, we display a simple message.
			AjpMiscellaneousTools.error( "Unable to get the compilation errors." );
		}

		// Now we get the class file.
		File classFile = new File(	AjpFileTools.classesScriptFolder()
										+ AjpFileTools.FILE_SEPARATOR + sourceFile.getName().substring( 0, sourceFile.getName().length() - ".java".length() ) + ".class" );
	
		loadClassFile( classFile );
	}
	
	/**
	 * Loads the given class file, supposing it is in the right directory.
	 * 
	 * @param classFile The class file to load.
	 */
	public void loadClassFile( File classFile ) {
		// Now we create a class loader.
		URLClassLoader classLoader;
		
		try {
			// We create a new class loader.
			classLoader = URLClassLoader.newInstance( new URL[] { new URL ( "file:" + AjpFileTools.classesFolder().getPath() + "/" ) } );
			
		} catch (MalformedURLException e) {
			// If we could not, we display an error and return.
			AjpMiscellaneousTools.error( "Unable to instantiate a class loader." );
			return;
		}
		
		Class<?> newClass;
		
		try {
			// We load the new class.
			newClass = classLoader.loadClass( "org.lozi.ajp.ajp.scripts." + classFile.getName().substring( 0, classFile.getName().length() - ".class".length() ) );
			//newClass = Class.forName( "org.lozi.ajp.ajp.Example1" );
		} catch (ClassNotFoundException e) {
			// If we could not, we display an error and return.
			AjpMiscellaneousTools.error( "Unable to load the new class." + e.getMessage() );
			return;
		}
				
		Object script;
		
		try {
			// We create an instance of this class.
			script = newClass.newInstance();
			
			// Then we set our script to the newly created object.
			this.script = ( AjpUserScript )script;
		} catch ( Exception ex ) {
			AjpMiscellaneousTools.error( "Unable to instantiate the new class." + ex.getMessage() );
			return;
		}
		
		// Hey! Did this *** user try to load a class which is not an AjpUserScript?
		if( ! ( script instanceof org.lozi.ajp.ajp.scripts.AjpUserScript ) ) {
			// **** off!
			AjpMiscellaneousTools.error( "This class does not implement AjpUserScript! " + script.getClass().getName() );
			// Such a mofo >_<
			return;
		}
	}
	
	/**
	 * Starts the execution.
	 */
	public void start () {
		// Pretty self explanatory...
		executionThread.run();
	}
	
	/**
	 * Stops the execution.
	 */
	public void stop () {
		for( AjpUserWindow frame : windowList )
			frame.close();
	}
	
	/**
	 * This class
	 * 
	 * @author Jean-Pierre Lozi - mailto:jean-pierre@lozi.org
	 */
	public class RunnableExecution implements Runnable {

		/**
		 * This function runs the thread.
		 */
		public void run() {
			script.start( plotter );
		}
	}

	
	/**
	 * Adds a frame.
	 * 
	 * @param frame The frame to add.
	 */
	public static void addFrame ( AjpUserWindow frame ) {
		windowList.add( frame );
	}
}